// /******************************************************************************
//  * File: XRHandTrackingMeshMSFTVisualizer.cs
//  * Copyright (c) 2025 Qualcomm Technologies, Inc. and/or its subsidiaries. All rights reserved.
//  *
//  * Confidential and Proprietary - Qualcomm Technologies, Inc.
//  *
//  ******************************************************************************/

using UnityEngine;
using UnityEngine.XR.OpenXR;
using QCHT.Interactions.Core;

namespace QCHT.Interactions.Hands
{
    public class XRHandTrackingMeshMSFTVisualizer : MonoBehaviour
    {
        [SerializeField] private XrHandedness handedness;

        public XrHandedness Handedness
        {
            get => handedness;
            set => handedness = value;
        }

        [SerializeField] private Material material;

        public Material Material
        {
            get => material;
            set => material = value;
        }

        private HandTrackingMeshMSFTFeature _feature;
        private MeshFilter _meshFilter;
        private MeshRenderer _meshRenderer;

        private void Awake()
        {
            var meshFeature = OpenXRSettings.Instance.GetFeature<HandTrackingMeshMSFTFeature>();
            if (meshFeature == null || !meshFeature.enabled)
            {
                Debug.LogWarning("[XRHandTrackingMeshMSFTVisualizer:Awake] " +
                                 "Hand Tracking Mesh feature is not activated in OpenXR settings");
                enabled = false;
                return;
            }

            if (!meshFeature.IsHandTrackingMeshMSFTSupported())
            {
                Debug.LogWarning("[XRHandTrackingMeshMSFTVisualizer:Awake] " +
                                 "XR_MSFT_hand_tracking_mesh extension is not supported on this runtime");
                enabled = false;
                return;
            }

            if (!meshFeature.TryGetHandTrackingMeshMSFT(out var mesh))
            {
                Debug.LogWarning("[XRHandTrackingMeshMSFTVisualizer:Awake] Failed to construct mesh");
                return;
            }

            _feature = meshFeature;

            _meshFilter = gameObject.AddComponent<MeshFilter>();

            if (_meshFilter == null)
            {
                enabled = false;
                return;
            }

            _meshRenderer = gameObject.AddComponent<MeshRenderer>();

            if (_meshRenderer == null)
            {
                enabled = false;
                return;
            }

            _meshRenderer.material = material;
            _meshFilter.mesh = mesh;
        }

        private void OnEnable()
        {
            Application.onBeforeRender += UpdateHandMesh;
        }

        private void OnDisable()
        {
            Application.onBeforeRender -= UpdateHandMesh;
        }

        private void UpdateHandMesh()
        {
            if (_feature == null)
            {
                return;
            }

            _feature.UpdateHandMesh(handedness, _meshFilter.mesh, out var pose);

            XROriginUtility.TransformPose(ref pose);

            transform.position = pose.position;
            transform.rotation = pose.rotation;
        }
    }
}