// /******************************************************************************
//  * File: XRHandTrackingMeshFBVisualizer.cs
//  * Copyright (c) 2025 Qualcomm Technologies, Inc. and/or its subsidiaries. All rights reserved.
//  *
//  * Confidential and Proprietary - Qualcomm Technologies, Inc.
//  *
//  ******************************************************************************/

using System;
using System.Collections.Generic;
using UnityEngine;
using QCHT.Interactions.Core;
using QCHT.Interactions.Core.OpenXR;
using UnityEngine.XR.OpenXR;

#if XR_HANDS
using UnityEngine.XR.Hands;
using XRHandTrackingSubsystem = UnityEngine.XR.Hands.XRHandSubsystem;
#endif

namespace QCHT.Interactions.Hands
{
    public class XRHandTrackingMeshFBVisualizer : MonoBehaviour
    {
        [SerializeField] private XrHandedness handedness;

        public XrHandedness Handedness
        {
            get => handedness;
            set => handedness = value;
        }
        
        [SerializeField] private Material material;

        public Material Material
        {
            get => material;
            set => material = value;
        }
        
        private XRHandTrackingSubsystem _handTrackingSubsystem;
        private SkinnedMeshRenderer _skRenderer;

        private void Awake()
        {
            var meshFeature = OpenXRSettings.Instance.GetFeature<HandTrackingMeshFBFeature>();
            if (meshFeature == null || !meshFeature.enabled)
            {
                Debug.LogWarning("[XRHandTrackingMeshFBVisualizer:Awake] " +
                                 "Hand Tracking Mesh feature is not activated in OpenXR settings");
                enabled = false;
                return;
            }

            if (!meshFeature.IsHandTrackingMeshFBSupported())
            {
                Debug.LogWarning("[XRHandTrackingMeshFBVisualizer:Awake] " +
                                 "XR_FB_hand_tracking_mesh extension is not supported on this runtime");
                enabled = false;
                return;
            }

            if (!meshFeature.TryGetHandTrackingMeshFB(handedness, out var mesh, out var bones))
            {
                Debug.LogWarning("[XRHandTrackingMeshFBVisualizer:Awake] " +
                                 "Failed to retrieved {handedness} hand mesh");
                enabled = false;
                return;
            }

            _skRenderer = gameObject.AddComponent<SkinnedMeshRenderer>();

            if (_skRenderer == null)
            {
                enabled = false;
                return;
            }

            _skRenderer.material = material;
            _skRenderer.sharedMesh = mesh;
            _skRenderer.bones = bones;
        }

        private void OnEnable()
        {
            Application.onBeforeRender += UpdateMesh;
        }

        private void OnDisable()
        {
            Application.onBeforeRender -= UpdateMesh;
        }

        private void UpdateMesh()
        {
            if (_handTrackingSubsystem == null)
            {
                var handSubsystem = new List<XRHandTrackingSubsystem>();
                SubsystemManager.GetSubsystems(handSubsystem);
                if (handSubsystem.Count > 0)
                {
                    _handTrackingSubsystem = handSubsystem[0];
                }
            }

            if (_handTrackingSubsystem == null)
            {
                return;
            }

            UpdateHandJoints();
        }

#if XR_HANDS
        private void UpdateHandJoints()
        {
            var hand = handedness == XrHandedness.XR_HAND_LEFT
                ? _handTrackingSubsystem.leftHand
                : _handTrackingSubsystem.rightHand;

            for (var i = 0; i < _skRenderer.bones.Length; i++)
            {
                var bone = _skRenderer.bones[i];
                var joint = hand.GetJoint(GetXRHandJointIDFromXrHandJointExt((XrHandJointEXT)i));

                if (joint.TryGetPose(out var pose))
                {
                    XROriginUtility.TransformPose(ref pose, true);

                    bone.position = pose.position;
                    bone.rotation = pose.rotation;
                }
            }
        }

        private static XRHandJointID GetXRHandJointIDFromXrHandJointExt(XrHandJointEXT handJointExt) =>
            handJointExt switch
            {
                XrHandJointEXT.XR_HAND_JOINT_PALM_EXT => XRHandJointID.Palm,
                XrHandJointEXT.XR_HAND_JOINT_WRIST_EXT => XRHandJointID.Wrist,
                XrHandJointEXT.XR_HAND_JOINT_THUMB_METACARPAL_EXT => XRHandJointID.ThumbMetacarpal,
                XrHandJointEXT.XR_HAND_JOINT_THUMB_PROXIMAL_EXT => XRHandJointID.ThumbProximal,
                XrHandJointEXT.XR_HAND_JOINT_THUMB_DISTAL_EXT => XRHandJointID.ThumbDistal,
                XrHandJointEXT.XR_HAND_JOINT_THUMB_TIP_EXT => XRHandJointID.ThumbTip,
                XrHandJointEXT.XR_HAND_JOINT_INDEX_METACARPAL_EXT => XRHandJointID.IndexMetacarpal,
                XrHandJointEXT.XR_HAND_JOINT_INDEX_PROXIMAL_EXT => XRHandJointID.IndexProximal,
                XrHandJointEXT.XR_HAND_JOINT_INDEX_INTERMEDIATE_EXT => XRHandJointID.IndexIntermediate,
                XrHandJointEXT.XR_HAND_JOINT_INDEX_DISTAL_EXT => XRHandJointID.IndexDistal,
                XrHandJointEXT.XR_HAND_JOINT_INDEX_TIP_EXT => XRHandJointID.IndexTip,
                XrHandJointEXT.XR_HAND_JOINT_MIDDLE_METACARPAL_EXT => XRHandJointID.MiddleMetacarpal,
                XrHandJointEXT.XR_HAND_JOINT_MIDDLE_PROXIMAL_EXT => XRHandJointID.MiddleProximal,
                XrHandJointEXT.XR_HAND_JOINT_MIDDLE_INTERMEDIATE_EXT => XRHandJointID.MiddleIntermediate,
                XrHandJointEXT.XR_HAND_JOINT_MIDDLE_DISTAL_EXT => XRHandJointID.MiddleDistal,
                XrHandJointEXT.XR_HAND_JOINT_MIDDLE_TIP_EXT => XRHandJointID.MiddleTip,
                XrHandJointEXT.XR_HAND_JOINT_RING_METACARPAL_EXT => XRHandJointID.RingMetacarpal,
                XrHandJointEXT.XR_HAND_JOINT_RING_PROXIMAL_EXT => XRHandJointID.RingProximal,
                XrHandJointEXT.XR_HAND_JOINT_RING_INTERMEDIATE_EXT => XRHandJointID.RingIntermediate,
                XrHandJointEXT.XR_HAND_JOINT_RING_DISTAL_EXT => XRHandJointID.RingDistal,
                XrHandJointEXT.XR_HAND_JOINT_RING_TIP_EXT => XRHandJointID.RingTip,
                XrHandJointEXT.XR_HAND_JOINT_LITTLE_METACARPAL_EXT => XRHandJointID.LittleMetacarpal,
                XrHandJointEXT.XR_HAND_JOINT_LITTLE_PROXIMAL_EXT => XRHandJointID.LittleProximal,
                XrHandJointEXT.XR_HAND_JOINT_LITTLE_INTERMEDIATE_EXT => XRHandJointID.LittleIntermediate,
                XrHandJointEXT.XR_HAND_JOINT_LITTLE_DISTAL_EXT => XRHandJointID.LittleDistal,
                XrHandJointEXT.XR_HAND_JOINT_LITTLE_TIP_EXT => XRHandJointID.LittleTip,
                _ => throw new ArgumentOutOfRangeException(nameof(handJointExt), handJointExt, null)
            };
#else
        private void UpdateHandJoints()
        {
            var joints = handedness == XrHandedness.XR_HAND_LEFT
                ? _handTrackingSubsystem.LeftHand.Joints
                : _handTrackingSubsystem.RightHand.Joints;

            for (var i = 0; i < _skRenderer.bones.Length; i++)
            {
                var bone = _skRenderer.bones[i];
                var joint = joints[i];

                XROriginUtility.TransformPose(ref joint, true);

                bone.position = joint.position;
                bone.rotation = joint.rotation;
            }
        }
#endif
    }
}