// /******************************************************************************
//  * File: XRHandTrackingQCHTProvider.cs
//  * Copyright (c) 2023 Qualcomm Technologies, Inc. and/or its subsidiaries. All rights reserved.
//  *
//  * Confidential and Proprietary - Qualcomm Technologies, Inc.
//  *
//  ******************************************************************************/

using Unity.Profiling;
using UnityEngine;
using QCHT.Interactions.Core.OpenXR;

namespace QCHT.Interactions.Core
{
    public class XRHandTrackingQCHTProvider : XRHandTrackingSubsystem.Provider
    {
        public const string ID = "Qualcomm-HandTracking-Native";

        private XrPosef[] _leftPoses = new XrPosef[(int)XrHandJoint.XR_HAND_JOINT_MAX];
        private XrPosef[] _rightPoses = new XrPosef[(int)XrHandJoint.XR_HAND_JOINT_MAX];

        private static readonly ProfilerMarker s_tryUpdateHandDataMarker =
            new("[QCHT] XRHandTrackingQCHTProvider.TryUpdateHandData");

        private static readonly ProfilerMarker s_tryLocateHandJointsMarker =
            new("[QCHT] XRHandTrackingQCHTProvider.TryUpdateHandData.TryLocateHandJoints");

        public override HandTrackingStatus TryUpdateHandData(ref XRHandTrackingSubsystem.Hand hand)
        {
            using (s_tryUpdateHandDataMarker.Auto())
            {
                var isLeft = hand._handedness == XrHandedness.XR_HAND_LEFT;
                ref var poses = ref isLeft ? ref _leftPoses : ref _rightPoses;
                var handSide = isLeft ? XrHandEXT.XR_HAND_LEFT : XrHandEXT.XR_HAND_RIGHT;

                XrResult result;

                using (s_tryLocateHandJointsMarker.Auto())
                {
                    result = HandTrackingOpenXRPlugin.TryLocateHandJoints(handSide, ref hand._isTracked, poses,
                        hand._radiuses);
                }

                // In this case just return idle status and wait for next frames
                if (result == XrResult.XR_ERROR_TIME_INVALID)
                {
                    return HandTrackingStatus.Idle;
                }

                if (result < XrResult.XR_SUCCESS)
                {
                    return HandTrackingStatus.Error;
                }

                for (var i = 0; i < (int)XrHandJoint.XR_HAND_JOINT_MAX; i++)
                {
                    hand._joints[i] = poses[i].ToPose();
                }

                hand._rootPose = hand._joints[(int)XrHandJoint.XR_HAND_JOINT_WRIST];
            }

            return HandTrackingStatus.Running;
        }

        [RuntimeInitializeOnLoadMethod(RuntimeInitializeLoadType.SubsystemRegistration)]
        private static void RegisterDescriptor()
        {
            XRHandTrackingSubsystem.Descriptor.Create(new XRHandTrackingSubsystem.Descriptor.Cinfo
            {
                id = ID,
                providerType = typeof(XRHandTrackingQCHTProvider)
            });
        }
    }
}