// /******************************************************************************
//  * File: XRHandTrackingSimulationLoader.cs
//  * Copyright (c) 2023 Qualcomm Technologies, Inc. and/or its subsidiaries. All rights reserved.
//  *
//  * Confidential and Proprietary - Qualcomm Technologies, Inc.
//  *
//  ******************************************************************************/

using System;
using System.Collections.Generic;
using UnityEngine;

namespace QCHT.Interactions.Core
{
    public class XRHandTrackingSimulationLoader : MonoBehaviour
    {
        private void OnEnable()
        {
            var list = new List<XRHandTrackingSubsystem>();
            SubsystemManager.GetSubsystems(list);
            foreach (var subsystem in list)
            {
                subsystem?.Start();
            }
        }

        private void OnDisable()
        {
            var list = new List<XRHandTrackingSubsystem>();
            SubsystemManager.GetSubsystems(list);
            foreach (var subsystem in list)
            {
                subsystem.Stop();
            }
        }

        private static readonly List<XRHandTrackingSubsystem.Descriptor> s_handTrackingSubsystemDescriptors = new();

#if UNITY_EDITOR || UNITY_STANDALONE
        [RuntimeInitializeOnLoadMethod(RuntimeInitializeLoadType.SubsystemRegistration)]
#endif
        public static void CreateHandTrackingSimulator()
        {
            if (!XRHandTrackingSimulationSettings.Instance.enabled)
                return;

            XRHandTrackingSubsystem.Descriptor.Create(new XRHandTrackingSubsystem.Descriptor.Cinfo
            {
                id = XRHandTrackingSimulationProvider.ID,
                providerType = typeof(XRHandTrackingSimulationProvider)
            });

            CreateSubsystem<XRHandTrackingSubsystem.Descriptor, XRHandTrackingSubsystem>(
                s_handTrackingSubsystemDescriptors, XRHandTrackingSimulationProvider.ID);

            var gameObject = new GameObject("Hand Tracking Simulator");
            gameObject.AddComponent<XRHandTrackingSimulationLoader>();
            DontDestroyOnLoad(gameObject);
        }

        private static ISubsystem CreateSubsystem<TDescriptor, TSubsystem>(List<TDescriptor> descriptors, string id)
            where TDescriptor : ISubsystemDescriptor where TSubsystem : ISubsystem
        {
            if (descriptors == null)
                throw new ArgumentNullException(nameof(descriptors));

            SubsystemManager.GetSubsystemDescriptors<TDescriptor>(descriptors);

            if (descriptors.Count <= 0)
                return null;

            foreach (var descriptor in descriptors)
                if (string.Compare(descriptor.id, id, StringComparison.OrdinalIgnoreCase) == 0)
                    return descriptor.Create();

            return null;
        }
    }
}