// /******************************************************************************
//  * File: HandTrackingMeshMSFTFeature.cs
//  * Copyright (c) 2024 Qualcomm Technologies, Inc. and/or its subsidiaries. All rights reserved.
//  *
//  * Confidential and Proprietary - Qualcomm Technologies, Inc.
//  *
//  ******************************************************************************/

using System;
using System.Runtime.InteropServices;
using QCHT.Interactions.Core.OpenXR;
using UnityEngine;

#if USING_SNAPDRAGON_SPACES_SDK
using Qualcomm.Snapdragon.Spaces;
using XrPosef = QCHT.Interactions.Core.OpenXR.XrPosef;
using XrResult = QCHT.Interactions.Core.OpenXR.XrResult;
using XrStructureType = QCHT.Interactions.Core.OpenXR.XrStructureType;
#else
using UnityEngine.XR.OpenXR.Features;
#endif

#if UNITY_EDITOR
using UnityEditor;
using UnityEditor.XR.OpenXR.Features;
#endif

namespace QCHT.Interactions.Core
{
#if UNITY_EDITOR
    [OpenXRFeature(
        FeatureId = FeatureId,
        UiName = FeatureName,
        Desc = FeatureDescription,
        Company = "Qualcomm",
        BuildTargetGroups = new[] { BuildTargetGroup.Android },
#if !USING_SNAPDRAGON_SPACES_SDK && USING_SNAPDRAGON_CUSTOM_OPENXR_LOADER
        CustomRuntimeLoaderBuildTargets = new[] { BuildTarget.Android },
#endif
        DocumentationLink = "",
        OpenxrExtensionStrings = FeatureExtensions,
        Version = "4.1.13",
        Required = false,
        Category = FeatureCategory.Feature)]
#endif
#if USING_SNAPDRAGON_SPACES_SDK
    public partial class HandTrackingMeshMSFTFeature : SpacesOpenXRFeature
#else
    public partial class HandTrackingMeshMSFTFeature : OpenXRFeature
#endif
    {
#if USING_SNAPDRAGON_SPACES_SDK
        public const string FeatureId = "com.qualcomm.snapdragon.spaces.handtracking.mesh.msft";
#if SPACES_0_23_2_TO_0_24_0 || SPACES_0_26_0_OR_NEWER
        internal override bool RequiresRuntimeCameraPermissions => false;
#endif
#else
        public const string FeatureId = "com.qualcomm.snapdragon.handtracking.mesh.msft";
#endif
        public const string FeatureDescription = "Allow to retreive MSFT hand mesh from runtime." +
                                                 "This extension needs the XR_EXT_hand_tracking extension enabled as well.";

        public const string FeatureName = "Hand Tracking Mesh MSFT";
        public const string FeatureExtensions = "XR_MSFT_hand_tracking_mesh";

        private int[] _tmpIndices;
        private Vector3[] _tmpVertices;
        private Vector3[] _tmpNormals;

        private int _maxHandMeshIndexCount = 0;
        private int _maxHandMeshVertexCount = 0;

        protected override IntPtr HookGetInstanceProcAddr(IntPtr func)
        {
            var result = base.HookGetInstanceProcAddr(func);

#if !UNITY_EDITOR && UNITY_ANDROID
            return HandTrackingOpenXRPlugin.HookGetInstanceProcAddr(func);
#endif
            return result;
        }

        public bool IsHandTrackingMeshMSFTSupported()
        {
#if !UNITY_EDITOR && UNITY_ANDROID
            return HandTrackingOpenXRPlugin.IsHandTrackingMeshMSFTSupported(ref _maxHandMeshIndexCount,
                ref _maxHandMeshVertexCount);
#endif

            Debug.LogWarning("[HandTrackingMeshMSFTFeature:IsHandTrackingMeshMSFTSupported] Platform not supported");
            return false;
        }

        public bool TryGetHandTrackingMeshMSFT(out Mesh mesh)
        {
            if (_maxHandMeshVertexCount == 0 || _maxHandMeshIndexCount == 0)
            {
                Debug.LogWarning("[HandTrackingMeshMSFTFeature:TryGetHandTrackingMeshMSFT] " +
                                 "Vertex or indices counts are zero. " +
                                 "Call IsHandTrackingMeshMSFTSupported before calling this method.");
                mesh = null;
                return false;
            }

            _tmpVertices = new Vector3[_maxHandMeshVertexCount];
            _tmpIndices = new int[_maxHandMeshIndexCount];
            _tmpNormals = new Vector3[_maxHandMeshVertexCount];

            mesh = new Mesh()
            {
                vertices = new Vector3[_maxHandMeshVertexCount],
                triangles = new int[_maxHandMeshIndexCount],
                normals = new Vector3[_maxHandMeshVertexCount],
            };

            return true;
        }

        public void UpdateHandMesh(XrHandedness handedness, Mesh mesh, out Pose pose)
        {
            var handMeshMsft = new XrHandMeshMSFT() { type = XrStructureType.XR_TYPE_HAND_MESH_MSFT };

            handMeshMsft.indexBuffer.indexCapacityInput = _maxHandMeshIndexCount;
            handMeshMsft.indexBuffer.indices =
                Marshal.AllocHGlobal(Marshal.SizeOf(typeof(int)) * _maxHandMeshIndexCount);

            handMeshMsft.vertexBuffer.vertexCapacityInput = _maxHandMeshVertexCount;
            handMeshMsft.vertexBuffer.vertices =
                Marshal.AllocHGlobal(Marshal.SizeOf(typeof(XrHandMeshVertexMSFT)) * _maxHandMeshVertexCount);

            var xrPose = new XrPosef();
            var handExt = handedness == XrHandedness.XR_HAND_LEFT ? XrHandEXT.XR_HAND_LEFT : XrHandEXT.XR_HAND_RIGHT;

            var result = HandTrackingOpenXRPlugin.TryGetHandMeshMSFT(handExt, ref handMeshMsft, ref xrPose);

            if (result < XrResult.XR_SUCCESS || !handMeshMsft.isActive)
            {
                pose = Pose.identity;
                return;
            }

            pose = xrPose.ToPose();

            for (var i = 0; i < _tmpVertices.Length; i++)
            {
                var vertex = Marshal.PtrToStructure<XrHandMeshVertexMSFT>(handMeshMsft.vertexBuffer.vertices +
                                                                          Marshal.SizeOf(typeof(XrHandMeshVertexMSFT)) *
                                                                          i);
                _tmpVertices[i] = vertex.position.ToVector3();
                _tmpNormals[i] = vertex.normal.ToVector3();
            }

            mesh.vertices = _tmpVertices;
            mesh.normals = _tmpNormals;

            for (var i = 0; i < _tmpIndices.Length; i++)
            {
                var index = Marshal.PtrToStructure<int>(handMeshMsft.indexBuffer.indices +
                                                        Marshal.SizeOf(typeof(int)) * i);
                _tmpIndices[i] = index;
            }

            mesh.triangles = _tmpIndices;

            Marshal.FreeHGlobal(handMeshMsft.indexBuffer.indices);
            Marshal.FreeHGlobal(handMeshMsft.vertexBuffer.vertices);
        }
    }
}