// /******************************************************************************
//  * File: HandTrackingMeshFBFeature.cs
//  * Copyright (c) 2024 Qualcomm Technologies, Inc. and/or its subsidiaries. All rights reserved.
//  *
//  * Confidential and Proprietary - Qualcomm Technologies, Inc.
//  *
//  ******************************************************************************/

using System;
using System.Runtime.InteropServices;
using System.Text;
using QCHT.Interactions.Core.OpenXR;
using UnityEngine;

#if UNITY_EDITOR
using UnityEditor;
using UnityEditor.XR.OpenXR.Features;
#endif

#if USING_SNAPDRAGON_SPACES_SDK
using Qualcomm.Snapdragon.Spaces;
using XrStructureType = QCHT.Interactions.Core.OpenXR.XrStructureType;
using XrVector2f = QCHT.Interactions.Core.OpenXR.XrVector2f;
using XrVector3f = QCHT.Interactions.Core.OpenXR.XrVector3f;
using XrPosef = QCHT.Interactions.Core.OpenXR.XrPosef;
using XrResult = QCHT.Interactions.Core.OpenXR.XrResult;
#else
using UnityEngine.XR.OpenXR.Features;
#endif

namespace QCHT.Interactions.Core
{
#if UNITY_EDITOR
    [OpenXRFeature(
        FeatureId = FeatureId,
        UiName = FeatureName,
        Desc = FeatureDescription,
        Company = "Qualcomm",
        BuildTargetGroups = new[] { BuildTargetGroup.Android },
#if !USING_SNAPDRAGON_SPACES_SDK && USING_SNAPDRAGON_CUSTOM_OPENXR_LOADER
        CustomRuntimeLoaderBuildTargets = new[] { BuildTarget.Android },
#endif
        DocumentationLink = "",
        OpenxrExtensionStrings = FeatureExtensions,
        Version = "4.1.13",
        Required = false,
        Category = FeatureCategory.Feature)]
#endif
#if USING_SNAPDRAGON_SPACES_SDK
    public partial class HandTrackingMeshFBFeature : SpacesOpenXRFeature
#else
    public partial class HandTrackingMeshFBFeature : OpenXRFeature
#endif
    {
#if USING_SNAPDRAGON_SPACES_SDK
        public const string FeatureId = "com.qualcomm.snapdragon.spaces.handtracking.mesh.fb";

#if SPACES_0_23_2_TO_0_24_0 || SPACES_0_26_0_OR_NEWER
        internal override bool RequiresRuntimeCameraPermissions => false;
#endif
#else
        public const string FeatureId = "com.qualcomm.snapdragon.handtracking.mesh.fb";
#endif

        public const string FeatureDescription = "Allow to retreive FB hand mesh from runtime." +
                                                 "This extension needs the XR_EXT_hand_tracking extension enabled as well.";

        public const string FeatureName = "Hand Tracking Mesh FB";
        public const string FeatureExtensions = "XR_FB_hand_tracking_mesh";

        protected override IntPtr HookGetInstanceProcAddr(IntPtr func)
        {
            var result = base.HookGetInstanceProcAddr(func);

#if !UNITY_EDITOR && UNITY_ANDROID
            return HandTrackingOpenXRPlugin.HookGetInstanceProcAddr(func);
#endif
            return result;
        }

        public bool IsHandTrackingMeshMSFTSupported(ref int maxHandMeshIndexCount, ref int maxHandMeshVertexCount)
        {
#if !UNITY_EDITOR && UNITY_ANDROID
            return HandTrackingOpenXRPlugin.IsHandTrackingMeshMSFTSupported(ref maxHandMeshIndexCount,
                ref maxHandMeshVertexCount);
#endif
            return false;
        }

        public bool IsHandTrackingMeshFBSupported()
        {
#if !UNITY_EDITOR && UNITY_ANDROID
            return HandTrackingOpenXRPlugin.IsHandTrackingMeshFBSupported();
#endif
            Debug.LogWarning("[HandTrackingMeshFBFeature:IsHandTrackingMeshFBSupported] Platform not supported");

            return false;
        }

        public bool TryGetHandTrackingMeshFB(XrHandedness handedness, out Mesh mesh, out Transform[] bones)
        {
            var handExt = handedness == XrHandedness.XR_HAND_LEFT ? XrHandEXT.XR_HAND_LEFT : XrHandEXT.XR_HAND_RIGHT;
            var handMeshFb = new XrHandTrackingMeshFB { type = XrStructureType.XR_TYPE_HAND_TRACKING_MESH_FB };

#if !UNITY_EDITOR && UNITY_ANDROID
            if (HandTrackingOpenXRPlugin.TryGetHandMeshFB(handExt, ref handMeshFb) != XrResult.XR_SUCCESS)
            {
                Debug.Log($"[HandTrackingMeshFBFeature:Awake] Failed to retrieved {handExt} mesh");
                
                mesh = null;
                bones = null;
                
                return false;
            }
#endif

            handMeshFb.vertexCapacityInput = handMeshFb.vertexCountOutput;
            handMeshFb.indexCapacityInput = handMeshFb.indexCountOutput;
            handMeshFb.jointCapacityInput = handMeshFb.jointCountOutput;

            // Blend weights
            var boneWeights = new BoneWeight[handMeshFb.vertexCapacityInput];
            var blendWeights = new XrVector4f[handMeshFb.vertexCapacityInput];
            var blendWeightsHandle = GCHandle.Alloc(blendWeights, GCHandleType.Pinned);
            handMeshFb.vertexBlendWeights = blendWeightsHandle.AddrOfPinnedObject();

            // Blend indices
            var blendIndices = new XrVector4sFB[handMeshFb.vertexCapacityInput];
            var blendIndicesHandle = GCHandle.Alloc(blendIndices, GCHandleType.Pinned);
            handMeshFb.vertexBlendIndices = blendIndicesHandle.AddrOfPinnedObject();

            // Vertices
            var vertices = new Vector3[handMeshFb.vertexCapacityInput];
            var verticesNative = new XrVector3f[handMeshFb.vertexCapacityInput];
            var verticesHandle = GCHandle.Alloc(verticesNative, GCHandleType.Pinned);
            handMeshFb.vertexPositions = verticesHandle.AddrOfPinnedObject();

            // Indices
            var indices = new short[handMeshFb.indexCapacityInput];
            var triangles = new int[handMeshFb.indexCapacityInput];
            var indicesHandle = GCHandle.Alloc(indices, GCHandleType.Pinned);
            handMeshFb.indices = indicesHandle.AddrOfPinnedObject();

            // Normals
            var normals = new Vector3[handMeshFb.vertexCapacityInput];
            var normalsNative = new XrVector3f[handMeshFb.vertexCapacityInput];
            var normalsHandle = GCHandle.Alloc(normalsNative, GCHandleType.Pinned);
            handMeshFb.vertexNormals = normalsHandle.AddrOfPinnedObject();

            // UVs
            var uvs = new Vector2[handMeshFb.vertexCapacityInput];
            var uvsNative = new XrVector2f[handMeshFb.vertexCapacityInput];
            var uvsHandle = GCHandle.Alloc(uvsNative, GCHandleType.Pinned);
            handMeshFb.vertexUVs = uvsHandle.AddrOfPinnedObject();

            // Joints bind poses
            var bindPoses = new Matrix4x4[handMeshFb.jointCapacityInput];
            var jointBindPoses = new XrPosef[handMeshFb.jointCapacityInput];
            var jointBindPosesHandle = GCHandle.Alloc(jointBindPoses, GCHandleType.Pinned);
            handMeshFb.jointBindPoses = jointBindPosesHandle.AddrOfPinnedObject();

            // Joints radii
            var jointRadii = new float[handMeshFb.jointCapacityInput];
            var jointRadiiHandle = GCHandle.Alloc(jointRadii, GCHandleType.Pinned);
            handMeshFb.jointRadii = jointRadiiHandle.AddrOfPinnedObject();

            // Joints parents
            var jointParents = new XrHandJointEXT[handMeshFb.jointCapacityInput];
            var jointParentsHandle = GCHandle.Alloc(jointParents, GCHandleType.Pinned);
            handMeshFb.jointParents = jointParentsHandle.AddrOfPinnedObject();

#if !UNITY_EDITOR && UNITY_ANDROID
            if (HandTrackingOpenXRPlugin.TryGetHandMeshFB(handExt, ref handMeshFb) != XrResult.XR_SUCCESS)
            {
                Debug.Log($"[HandTrackingMeshFBFeature:Awake] Failed to retrieved {handExt} mesh");

                mesh = null;
                bones = null;

                blendWeightsHandle.Free();
                blendIndicesHandle.Free();
                verticesHandle.Free();
                indicesHandle.Free();
                normalsHandle.Free();
                uvsHandle.Free();
                jointBindPosesHandle.Free();
                jointRadiiHandle.Free();
                jointParentsHandle.Free();
                
                return false;
            }
#endif

            // Debug.Log($"[HandTrackingMeshFBFeature:ConstructHandMesh]\n" +
            //           $"Meshfb : \n" +
            //           $"Vertex count : {(int)handMeshFb.vertexCapacityInput} \n" +
            //           $"Triangles count : {(int)handMeshFb.indexCountOutput} \n" +
            //           $"Joints count : {(int)handMeshFb.jointCapacityInput} \n"
            // );

            bones = new Transform[handMeshFb.jointCountOutput];

            for (var i = 0; i < handMeshFb.jointCountOutput; i++)
            {
                var boneName = ((XrHandJointEXT)i).ToString();
                var goBone = new GameObject(boneName);

                bones[i] = goBone.transform;

                var sb = new StringBuilder();
                // sb.Append($"[OpenXRHandMesh:Awake] bone name = {boneName}");

                var pose = jointBindPoses[i].ToPose();
                bindPoses[i] = Matrix4x4.Inverse(Matrix4x4.TRS(pose.position, pose.rotation, Vector3.one));

                bones[i].localPosition = pose.position;
                bones[i].localRotation = pose.rotation;

                if (jointParents[i] >= XrHandJointEXT.XR_HAND_JOINT_PALM_EXT &&
                    jointParents[i] < XrHandJointEXT.XR_HAND_JOINT_LITTLE_TIP_EXT)
                {
                    // sb.Append($" -> parent = {jointParents[i]}");
                    goBone.transform.SetParent(bones[(int)jointParents[i]], true);
                }

                // sb.Append($" -> pose = {pose}");

                Debug.Log(sb.ToString());
            }

            for (var i = 0; i < handMeshFb.indexCountOutput; i += 3)
            {
                // Change winding to unity's counter-clockwise order
                triangles[i] = indices[i];
                triangles[i + 1] = indices[i + 2];
                triangles[i + 2] = indices[i + 1];
            }

            for (var i = 0; i < handMeshFb.vertexCountOutput; i++)
            {
                vertices[i] = verticesNative[i].ToVector3();
                normals[i] = normalsNative[i].ToVector3();

                // TODO: Remove Google specific uvs set once the shader is fixed
#if USE_GOOGLE_HAT
                uvs[i] = uvsNative[i].ToVector2();
#else
                // Inverts the y-axis as uvs starts bottom-left
                var v = uvsNative[i].ToVector2();
                v.y = 1.0f - v.y;
                uvs[i] = v;
#endif

                // Bone weights
                boneWeights[i].weight0 = blendWeights[i].x;
                boneWeights[i].weight1 = blendWeights[i].y;
                boneWeights[i].weight2 = blendWeights[i].z;
                boneWeights[i].weight3 = blendWeights[i].w;

                // Bone indices
                boneWeights[i].boneIndex0 = blendIndices[i].x;
                boneWeights[i].boneIndex1 = blendIndices[i].y;
                boneWeights[i].boneIndex2 = blendIndices[i].z;
                boneWeights[i].boneIndex3 = blendIndices[i].w;
            }

            mesh = new Mesh
            {
                vertices = vertices,
                normals = normals,
                triangles = triangles,
                boneWeights = boneWeights,
                bindposes = bindPoses,
                uv = uvs
            };

            mesh.RecalculateBounds();
            mesh.RecalculateTangents();

            blendWeightsHandle.Free();
            blendIndicesHandle.Free();
            verticesHandle.Free();
            indicesHandle.Free();
            normalsHandle.Free();
            uvsHandle.Free();
            jointBindPosesHandle.Free();
            jointRadiiHandle.Free();
            jointParentsHandle.Free();

            return true;
        }
    }
}