﻿// /******************************************************************************
//  * File: HandTrackingFeature.cs
//  * Copyright (c) 2023 Qualcomm Technologies, Inc. and/or its subsidiaries. All rights reserved.
//  *
//  * Confidential and Proprietary - Qualcomm Technologies, Inc.
//  *
//  ******************************************************************************/

using System;
using System.Collections.Generic;
using UnityEngine;
using QCHT.Interactions.Core.OpenXR;

#if UNITY_EDITOR
using UnityEditor;
using UnityEditor.XR.OpenXR.Features;
#endif

#if UNITY_ANDROID
using UnityEngine.Android;
#endif

#if USING_SNAPDRAGON_SPACES_SDK
using Qualcomm.Snapdragon.Spaces;
#else
using UnityEngine.XR.OpenXR.Features;
#endif

namespace QCHT.Interactions.Core
{
#if UNITY_EDITOR
    [OpenXRFeature(
        FeatureId = FeatureId,
        UiName = FeatureName,
        Desc = FeatureDescription,
        Company = "Qualcomm",
        BuildTargetGroups = new[] { BuildTargetGroup.Android },
#if !USING_SNAPDRAGON_SPACES_SDK && USING_SNAPDRAGON_CUSTOM_OPENXR_LOADER
        CustomRuntimeLoaderBuildTargets = new[] { BuildTarget.Android },
#endif
        DocumentationLink = "",
        OpenxrExtensionStrings = FeatureExtensions,
        Version = "4.1.13",
        Required = false,
        Category = FeatureCategory.Feature)]
#endif
#if USING_SNAPDRAGON_SPACES_SDK
    public partial class HandTrackingFeature : SpacesOpenXRFeature
#else
    public partial class HandTrackingFeature : OpenXRFeature
#endif
    {
#if USING_SNAPDRAGON_SPACES_SDK
        public const string FeatureId = "com.qualcomm.snapdragon.spaces.handtracking";
        public const string FeatureName = "Hand Tracking";
        public const string FeatureDescription = "Enables Hand Tracking feature on Snapdragon Spaces enabled devices.";
#if SPACES_0_23_2_TO_0_24_0 || SPACES_0_26_0_OR_NEWER 
        internal override bool RequiresRuntimeCameraPermissions => true;
#endif
#else
        public const string FeatureId = "com.qualcomm.snapdragon.handtracking";
        public const string FeatureName = "Qualcomm Hand Tracking";
        public const string FeatureDescription = "Enables Hand Tracking and gestures feature on Snapdragon enabled devices.";
#endif
        public const string FeatureExtensions = "XR_EXT_hand_tracking";

        [Tooltip("Should HaT subsystem automatically start after the OpenXR loader has started its subsystems?")]
        public bool AutoStart = true;

        private static readonly List<XRHandTrackingSubsystem.Descriptor> s_handTrackingSubsystemDescriptors = new();

        protected override bool OnInstanceCreate(ulong xrInstance)
        {
            return base.OnInstanceCreate(xrInstance);

#if UNITY_ANDROID && !UNITY_EDITOR
            var activity =
                new AndroidJavaClass("com.unity3d.player.UnityPlayer").GetStatic<AndroidJavaObject>("currentActivity");
            var runtimeChecker =
                new AndroidJavaClass("com.qualcomm.snapdragon.spaces.unityserviceshelper.RuntimeChecker");

            if (!runtimeChecker.CallStatic<bool>("CheckCameraPermissions", new object[] {activity}))
            {
                Debug.LogError("Snapdragon Spaces Services has no camera permissions! Hand Tracking feature disabled.");
                return false;
            }
#endif
        }

        protected override IntPtr HookGetInstanceProcAddr(IntPtr func)
        {
            var result = base.HookGetInstanceProcAddr(func);
         
#if !UNITY_EDITOR && UNITY_ANDROID
            return HandTrackingOpenXRPlugin.HookGetInstanceProcAddr(func);
#endif
            return result;
        }

        protected override void OnSubsystemCreate()
        {
            CreateSubsystem<XRHandTrackingSubsystem.Descriptor, XRHandTrackingSubsystem>(
                s_handTrackingSubsystemDescriptors, XRHandTrackingQCHTProvider.ID);
        }

        protected override void OnSubsystemStart()
        {
            if (AutoStart)
            {
                StartSubsystem<XRHandTrackingSubsystem>();
            }
        }

        protected override void OnSubsystemStop()
        {
            StopSubsystem<XRHandTrackingSubsystem>();
        }

        protected override void OnSubsystemDestroy()
        {
            DestroySubsystem<XRHandTrackingSubsystem>();
        }

        protected override void OnAppSpaceChange(ulong xrSpace)
        {
#if !UNITY_EDITOR && UNITY_ANDROID
            HandTrackingOpenXRPlugin.SetAppSpace(xrSpace);
#endif
        }

        public static string GetCurrentInteractionProfileForHand(XrHandedness handedness)
        {
            const string userPathHandLeft = "/user/hand/left";
            const string userPathHandRight = "/user/hand/right";
            var userPath = handedness == XrHandedness.XR_HAND_LEFT ? userPathHandLeft : userPathHandRight;
            return PathToString(GetCurrentInteractionProfile(userPath));
        }

        public static bool IsHandTrackingSupported()
        {
#if UNITY_EDITOR
            return true;
#endif
            return HandTrackingOpenXRPlugin.IsHandTrackingSupported();
        }
    }
}