// /******************************************************************************
//  * File: HandTrackingFeatureBuildHooks.cs
//  * Copyright (c) 2024 Qualcomm Technologies, Inc. and/or its subsidiaries. All rights reserved.
//  *
//  * Confidential and Proprietary - Qualcomm Technologies, Inc.
//  *
//  ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Reflection;
using QCHT.Interactions.Core;
using Unity.XR.Management.AndroidManifest.Editor;
using UnityEditor;
using UnityEditor.Build.Reporting;
using UnityEditor.XR.OpenXR.Features;
using UnityEngine.XR.OpenXR;
using UnityEngine.XR.OpenXR.Features;

#if XR_HANDS
using HandTrackingFeature = UnityEngine.XR.Hands.OpenXR.HandTracking;
#endif

namespace QCHT.Interactions.Android.Editor
{
    public class HandTrackingFeatureBuildHooks : OpenXRFeatureBuildHooks
    {
        public override int callbackOrder => 1;

        public override Type featureType => typeof(HandTrackingFeature);

        protected override void OnPreprocessBuildExt(BuildReport report)
        {
            // Ignored
        }

        protected override void OnPostprocessBuildExt(BuildReport report)
        {
            // Ignored
        }

        protected override void OnPostGenerateGradleAndroidProjectExt(string path)
        {
            // Ignored
        }

        protected override ManifestRequirement ProvideManifestRequirementExt()
        {
            var manifestRequirement = new ManifestRequirement
            {
                SupportedXRLoaders = new HashSet<Type>()
                {
                    typeof(OpenXRLoader)
                },

#if USING_SNAPDRAGON_CUSTOM_OPENXR_LOADER
                NewElements = new List<ManifestElement>()
                {
                    // <uses-permission android:name="com.qualcomm.qti.qxr.QXRServiceClientPermission" />
                    new ManifestElement
                    {
                        ElementPath = new List<string> {"manifest", "uses-permission" },
                        Attributes = new Dictionary<string, string>()
                        {   
                            {"name", "com.qualcomm.qti.qxr.QXRServiceClientPermission"}
                        }
                    },
                    // <package android:name="com.qualcomm.qti.openxrruntime" />
                    new ManifestElement()
                    {
                        ElementPath = new List<string> {"manifest", "queries", "package"},
                        Attributes = new Dictionary<string, string>()
                        {   
                            {"name", "com.qualcomm.qti.openxrruntime"}
                        }
                    }
                },
#endif

#if UNITY_ANDROID_XR
                NewElements = new List<ManifestElement>()
                {
                    // <property android:name="android.window.PROPERTY_ACTIVITY_STARTS_IN_IMMERSIVE_XR" android:value="true"/>
                    new ManifestElement()
                    {
                        ElementPath = new List<string> { "manifest", "application", "activity", "property" },
                        Attributes = new Dictionary<string, string>()
                        {
                            { "name", "android.window.PROPERTY_ACTIVITY_STARTS_IN_IMMERSIVE_XR" },
                            { "value", "true" }
                        }
                    },
                    // <category android:name="android.intent.category.XR" />
                    new ManifestElement()
                    {
                        ElementPath = new List<string>
                            { "manifest", "application", "activity", "intent-filter", "category" },
                        Attributes = new Dictionary<string, string>()
                        {
                            { "name", "android.intent.category.XR" }
                        }
                    },
                    // <uses-permission android:name="android.permission.HAND_TRACKING" />
                    new ManifestElement
                    {
                        ElementPath = new List<string> { "manifest", "uses-permission" },
                        Attributes = new Dictionary<string, string>()
                        {
                            { "name", "android.permission.HAND_TRACKING" }
                        }
                    },
                    // <uses-permission android:name="android.permission.CAMERA" />
                    new ManifestElement
                    {
                        ElementPath = new List<string> { "manifest", "uses-permission" },
                        Attributes = new Dictionary<string, string>()
                        {
                            { "name", "android.permission.CAMERA" }
                        }
                    },
                    //<uses-permission android:name="android.permission.XR_EXPERIMENTAL_FEATURES"/>
                    new ManifestElement
                    {
                        ElementPath = new List<string> { "manifest", "uses-permission" },
                        Attributes = new Dictionary<string, string>()
                        {
                            { "name", "android.permission.XR_EXPERIMENTAL_FEATURES" }
                        }
                    }
                },
#endif
            };

            // Gets Open XR profile string of every enabled OpenXRInteraction feature and checks if it contains "controller" in path. 
            // The profile constant seems to be named "profile" in every OpenXRInteraction features inherited classes.
            // It is not a OpenXRInteraction field though, so we try to get it by reflection.
            // It allows to check if a controller profile has been enabled in Open XR build settings
            // In case of a controller profile enabled, it will assume that the hand tracking is not required and can support both interactions (hand tracking + motion controllers)
            var required = true;
            var features = OpenXRSettings.GetSettingsForBuildTargetGroup(BuildTargetGroup.Android);
            foreach (var interactionFeature in features.GetFeatures<OpenXRInteractionFeature>())
            {
                if (!interactionFeature.enabled)
                {
                    continue;
                }

                var type = interactionFeature.GetType();
                const BindingFlags bindingFlags = BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Static;
                var fieldInfo = type.GetField("profile", bindingFlags);
                if (fieldInfo == null)
                {
                    continue;
                }

                if (fieldInfo.GetValue(interactionFeature) is string profile && profile.Contains("controller"))
                {
                    required = false;
                }
            }

            // <uses-feature android:name="qualcomm.software.handtracking" />
            manifestRequirement.NewElements.Add(new ManifestElement()
            {
                ElementPath = new List<string>() { "manifest", "uses-feature" },
                Attributes = new Dictionary<string, string>()
                {
                    { "name", "qualcomm.software.handtracking" },
                    { "required", required ? "true" : "false" }
                }
            });

            return manifestRequirement;
        }
    }
}