/*
 * Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
 * All rights reserved.
 * Confidential and Proprietary - Qualcomm Technologies, Inc.
 */

package com.qualcomm.snapdragon.spaces.helpers;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.ActivityNotFoundException;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.net.Uri;
import android.util.Log;

import com.qualcomm.snapdragon.spaces.R;
import com.qualcomm.qti.device.access.*;


import com.qualcomm.qti.openxr.compatibility.olderruntimecompatibility.OlderRuntimeCompatibility;
public class CheckSpacesServicesHelper
{
    private static final String TAG = "Spaces-CheckSpacesServicesHelper";

    private Activity _callingActivity;

    private static final String _servicesPackageId = "com.qualcomm.qti.spaces.services";
    private static final String _openXrRuntimePackageId = "com.qualcomm.qti.openxrruntime";

    public CheckSpacesServicesHelper(Activity activity) {
        _callingActivity = activity;
    }

    public boolean CheckSpacesServicesInstalled() {
        boolean runtimeExists = false;
        // Check for Spaces OpenXR runtime
        try {
            DeviceAccessManager dam = DeviceAccessManager.getInstance(null);
            _callingActivity.getPackageManager().getPackageInfo(dam.getRuntimePackageName(_callingActivity), 0);
            Log.v(TAG, "An OpenXR runtime compatible with Snapdragon Spaces was detected");
            runtimeExists = true;
        } catch (PackageManager.NameNotFoundException e) {
            Log.w(TAG, "An OpenXR runtime compatible with Snapdragon Spaces was NOT detected");
        }

        if (!runtimeExists) {
            Log.v(TAG, "An OpenXR runtime compatible with Snapdragon Spaces is NOT installed");
            showAlertDialog(_callingActivity.getString(R.string.services_not_installed),
                    _callingActivity.getString(R.string.please_install_services),
                    null,
                    null,
                    _callingActivity.getString(R.string.quit),
                    new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int id) {
                            _callingActivity.finish();
                        }
                    },
                    false);
        }

        return runtimeExists;
    }

    public boolean CheckSpacesServicesCompatible(){
        OlderRuntimeCompatibility _ORC = new OlderRuntimeCompatibility();

        OlderRuntimeCompatibility.Result r = _ORC.initializeCompatibilitySystem(_callingActivity.getApplicationContext());
        _ORC.shutdownCompatibilitySystem();
        if(r != OlderRuntimeCompatibility.Result.SUCCESS){
            Log.e(TAG, r.name());
            switch(r){
                case ERROR_APPLICATION_TOO_OLD_FOR_RUNTIME:
                    showAlertDialog(_callingActivity.getString(R.string.app_too_old),
                            _callingActivity.getString(R.string.app_too_old_desc),
                            null,
                            null,
                            _callingActivity.getString(R.string.quit),
                            new DialogInterface.OnClickListener() {
                                public void onClick(DialogInterface dialog, int id) {
                                    _callingActivity.finish();
                                }
                            },
                            false);
                    break;
                case ERROR_RUNTIME_TOO_OLD_FOR_APPLICATION:
                    showAlertDialog(_callingActivity.getString(R.string.services_too_old),
                            _callingActivity.getString(R.string.services_too_old_desc),
                            null,
                            null,
                            _callingActivity.getString(R.string.quit),
                            new DialogInterface.OnClickListener() {
                                public void onClick(DialogInterface dialog, int id) {
                                    _callingActivity.finish();
                                }
                            },
                            false);
                    break;
                case ERROR_RUNTIME_FAILURE:
                case ERROR_VALIDATION_FAILURE:
                    showAlertDialog(_callingActivity.getString(R.string.services_failure),
                            _callingActivity.getString(R.string.services_failure_desc),
                            null,
                            null,
                            _callingActivity.getString(R.string.quit),
                            new DialogInterface.OnClickListener() {
                                public void onClick(DialogInterface dialog, int id) {
                                    _callingActivity.finish();
                                }
                            },
                            false);
                    break;
            }
        }
        return r == OlderRuntimeCompatibility.Result.SUCCESS;
    }

    private void showAlertDialog(String title, String message, String positiveButtonText, DialogInterface.OnClickListener positiveEvent, String negativeButtonText, DialogInterface.OnClickListener negativeEvent) {
        showAlertDialog(title, message, positiveButtonText, positiveEvent, negativeButtonText, negativeEvent, true);
    }

    private void showAlertDialog(String title, String message, String positiveButtonText, DialogInterface.OnClickListener positiveEvent, String negativeButtonText, DialogInterface.OnClickListener negativeEvent, boolean cancelable) {
        boolean darkMode = (_callingActivity.getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK) == Configuration.UI_MODE_NIGHT_YES ? true : false;
        AlertDialog.Builder dialogBuilder = new AlertDialog.Builder(_callingActivity,
            darkMode ? AlertDialog.THEME_DEVICE_DEFAULT_DARK : AlertDialog.THEME_DEVICE_DEFAULT_LIGHT);
        dialogBuilder.setTitle(title);
        dialogBuilder.setMessage(message);
        if (positiveButtonText != null && positiveEvent != null) {
            dialogBuilder.setPositiveButton(positiveButtonText, positiveEvent);
        }
        if (negativeButtonText != null && negativeEvent != null) {
            dialogBuilder.setNegativeButton(negativeButtonText, negativeEvent);
        }
        dialogBuilder.setCancelable(cancelable);
        dialogBuilder.show();
    }
}
