using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using UnityEngine;
using UnityEngine.UI;

namespace com.nttqonoq.devices.android.mirzalibrary.samples
{
    public class SampleManager : MonoBehaviour
    {
        [SerializeField] private GameObject preview;
        [SerializeField] private ScrollRect scrollRect;
        [SerializeField] private Text logText;
        [SerializeField] private Text versionText;
        [SerializeField] private GameObject optionListDialogPrefab;
        
        private DateTime _lastLogUpdateTime;

        private SynchronizationContext _mainThreadContext;
        private MirzaPlugin _mirzaPlugin;

        private void Start()
        {
            _mainThreadContext = SynchronizationContext.Current;

#if UNITY_ANDROID && !UNITY_EDITOR
            // ライブラリ初期化
            _mirzaPlugin = new MirzaPlugin();
#endif
            _mirzaPlugin?.SetLogEnable(true);

            StartLibrary();
        }

        private void OnDestroy()
        {
            StopLibrary();
        }

        private void OnApplicationPause(bool pauseStatus)
        {
            if (pauseStatus)
            {
                Debug.Log("App paused in the background.");
                StopLibrary();
            }
            else
            {
                Debug.Log("App resumed in the foreground.");
                StartLibrary();
            }
        }

        private void StartLibrary()
        {
            if (_mirzaPlugin == null) return;

            _mirzaPlugin.OnServiceStateChanged += HandleServiceStateChanged;
            _mirzaPlugin.OnGlassStatusChanged += HandleGlassStatusChanged;
            _mirzaPlugin.OnBatteryLevelChanged += HandleBatteryLevelChanged;
            _mirzaPlugin.OnSpacesModeStatusChanged += HandleSpacesModeStatusChanged;
            _mirzaPlugin.OnDisplayStatusChanged += HandleDisplayStatusChange;
            _mirzaPlugin.OnPowerOffChanged += HandlePowerOffChanged;
            _mirzaPlugin.OnGlassTouchGestureStatusChanged += HandleGlassTouchGestureStatusChanged;


            _mirzaPlugin.StartMonitoring();

            // ライブラリバージョン
            versionText.text = $"version {_mirzaPlugin.GetVersion()}";
        }

        private void StopLibrary()
        {
            if (_mirzaPlugin == null) return;

            _mirzaPlugin.StopMonitoring();

            _mirzaPlugin.OnServiceStateChanged -= HandleServiceStateChanged;
            _mirzaPlugin.OnGlassStatusChanged -= HandleGlassStatusChanged;
            _mirzaPlugin.OnBatteryLevelChanged -= HandleBatteryLevelChanged;
            _mirzaPlugin.OnSpacesModeStatusChanged -= HandleSpacesModeStatusChanged;
            _mirzaPlugin.OnDisplayStatusChanged -= HandleDisplayStatusChange;
            _mirzaPlugin.OnPowerOffChanged -= HandlePowerOffChanged;
            _mirzaPlugin.OnGlassTouchGestureStatusChanged -= HandleGlassTouchGestureStatusChanged;
        }

        // サービス状態の更新ハンドラー
        private void HandleServiceStateChanged(ServiceState state)
        {
            AddLog($"Service State : {state}");
        }

        // デバイス状態の変化通知イベントハンドラー
        private void HandleGlassStatusChanged(GlassStatus status)
        {
            var log = $"[ Glass Status (Notify)] : {State.Success}";
            log += LogGlassStatus(status);
            AddLog(log);
        }

        // 電池残量の変化通知イベントハンドラー
        private void HandleBatteryLevelChanged(int level)
        {
            AddLog($"[ Battery Level (Notify) ] : {State.Success}\n{level}%");
        }


        // Spacesモードの変化通知イベントハンドラー
        private void HandleSpacesModeStatusChanged(SpacesModeStatus status)
        {
            AddLog($"[ Spaces Mode Status (Notify) ] : {State.Success}\n{status}");

            // Spacesモード状態によりPhone側のプレビュー表示・非表示を切り替える
            DisplayPreview(status);
        }

        // 画面表示状態の変化通知イベントハンドラー
        private void HandleDisplayStatusChange(DisplayStatus status)
        {
            AddLog($"[ Display Status (Notify) ] : {State.Success}\n{status}");
        }

        // 手動電源OFFの変更通知イベントハンドラー
        private void HandlePowerOffChanged()
        {
            AddLog($"[ Power Off (Notify) ] : {State.Success}");
        }

        // グラスジェスチャーの変化通知イベントハンドラー
        private void HandleGlassTouchGestureStatusChanged(GlassTouchGestureStatus status)
        {
            var log = $"[ Glass Touch Gesture Status (Notify) ] : {State.Success}";
            log += $"\nTouchType = {status.Type}";
            log += $"\nTouchOperation = {status.Operation}";
            log += $"\nCoordinate = ({status.XCoordinate},{status.YCoordinate})";
            log += $"\nMovement = {status.Movement}";
            AddLog(log, 50);
        }


        // ログをクリア
        public void OnLogTextClearButtonClick()
        {
            _mainThreadContext.Post(_ => { logText.text = ""; }, null);
        }

        // バージョン整合性を確認
        public void OnCheckVersionConsistencyButtonClick()
        {
            if (_mirzaPlugin != null)
            {
                var result = _mirzaPlugin?.CheckVersionConsistency();
                if (result == null) return;
                var log = $"[ Check Version Consistency ] : {result.State}";
                if (result.Data != null) log += $"\n{result.Data}";
                AddLog(log);
            }
            else
            {
                Debug.LogWarning("MirzaPlugin is not available.");
            }
        }


        // グラス状態を取得
        public void OnGlassStatusButtonClick()
        {
            if (_mirzaPlugin != null)
                _mirzaPlugin.GetGlassStatusAsync(result =>
                {
                    if (result == null) return;
                    var log = $"[ Glass Status (Read) ] : {result.State}";
                    if (result.Data != null) log += LogGlassStatus(result.Data);
                    AddLog(log);
                });
            else
                Debug.LogWarning("MirzaPlugin is not available.");
        }

        // 電池残量を取得
        public void OnBatteryButtonClick()
        {
            if (_mirzaPlugin != null)
                _mirzaPlugin.GetBatteryLevelAsync(result =>
                {
                    if (result == null) return;
                    var log = $"[ Battery Level (Read) ] : {result.State}";
                    if (result.State == State.Success) log += $"\n{result.Data} %";
                    AddLog(log);
                });
            else
                Debug.LogWarning("MirzaPlugin is not available.");
        }

        // 充電状態を取得
        public void OnChargeStatusButtonClick()
        {
            if (_mirzaPlugin != null)
                _mirzaPlugin.GetChargeStatusAsync(result =>
                {
                    if (result == null) return;
                    var log = $"[ Charge Status (Read) ] : {result.State}";
                    if (result.State == State.Success) log += $"\n{result.Data}";
                    AddLog(log);
                });
            else
                Debug.LogWarning("MirzaPlugin is not available.");
        }

        // Spacesモード状態を取得
        public void OnSpacesModeStatusButtonClick()
        {
            if (_mirzaPlugin != null)
                _mirzaPlugin.GetSpacesModeStatusAsync(result =>
                {
                    if (result == null) return;
                    var log = $"[ Spaces Mode Status (Read) ] : {result.State}";
                    if (result.State == State.Success) log += $"\n{result.Data}";
                    AddLog(log);

                    DisplayPreview(result.Data);
                });
            else
                Debug.LogWarning("MirzaPlugin is not available.");
        }


        // SPACESモードを有効化
        public void OnSpacesModeEnableButtonClick()
        {
            if (_mirzaPlugin != null)
                _mirzaPlugin.SpacesModeOnAsync(result =>
                {
                    if (result == null) return;
                    var log = $"[ Spaces Mode On (Write) ] : {result.State}";
                    if (result.ErrorMessage != null) log += $"\n{result.ErrorMessage}";
                    AddLog(log);
                });
            else
                Debug.LogWarning("MirzaPlugin is not available.");
        }

        // SPACESモードを無効化
        public void OnSpacesModeDisableButtonClick()
        {
            if (_mirzaPlugin != null)
                _mirzaPlugin.SpacesModeOffAsync(result =>
                {
                    if (result == null) return;
                    var log = $"[ Spaces Mode Off (Write) ] : {result.State}";
                    if (result.ErrorMessage != null) log += $"\n{result.ErrorMessage}";
                    AddLog(log);
                });
            else
                Debug.LogWarning("MirzaPlugin is not available.");
        }
        
        // マイク切替を実行
        public void OnSwitchMicrophoneButtonClick()
        {
            // 選択ダイアログ
            OptionListDialog.Show(
                "Switch Microphone",
                new List<(string, List<object>)>
                {
                    ("Wearing mic mode", Enum.GetValues(typeof(MicMode))
                        .Cast<object>()
                        .ToList()),
                    ("Front mic mode", Enum.GetValues(typeof(MicMode))
                        .Cast<object>()
                        .ToList()),
                    ("Mix mode", Enum.GetValues(typeof(MixMode))
                        .Cast<object>()
                        .ToList()),
                },
                (selections) =>
                {
                    var wearingMicMode = (MicMode)selections[0];
                    var frontMicMode = (MicMode)selections[1];
                    var mixMode = (MixMode)selections[2];
                    
                    Debug.Log($"Selected: {wearingMicMode}、{frontMicMode}、{mixMode}");

                    if (_mirzaPlugin != null)
                        _mirzaPlugin.SwitchMicrophoneAsync(wearingMicMode, frontMicMode, mixMode, result =>
                        {
                            if (result == null) return;
                            var log = $"[ Switch Microphone (Write) ] : {result.State}";
                            if (result.ErrorMessage != null) log += $"\n{result.ErrorMessage}";
                            if (result.Data != null) log += LogSwitchMicrophoneStatus(result.Data);
                            AddLog(log);
                            
                        });
                    else
                        Debug.LogWarning("MirzaPlugin is not available.");
                },
                optionListDialogPrefab
            );
        }
        
        // ユーザーがオプションを選択したときに呼び出されるメソッド
        private void OnSelectionComplete(List<string> selectedItems)
        {
            Debug.Log("選択されたColor: " + selectedItems[0]);
            Debug.Log("選択されたSize: " + selectedItems[1]);
            Debug.Log("選択されたShape: " + selectedItems[2]);
        }

        // MiRZAアプリへ遷移
        public void OnOpenMirzaAppButtonClick()
        {
            if (_mirzaPlugin != null)
            {
                var result = _mirzaPlugin?.TransitionToMirzaAppInSpacesMode();
                if (result == null) return;
                var log = $"[ Open MiRZA App ] : {result.State}";
                log += $"\nSpaces Mode {result.Data}";
                AddLog(log);
            }
            else
            {
                Debug.LogWarning("MirzaPlugin is not available.");
            }
        }

        // 端末のアプリ設定画面を表示
        public void OnOpenAppSettingsButtonClick()
        {
            try
            {
                using var unityPlayer = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
                var activity = unityPlayer.GetStatic<AndroidJavaObject>("currentActivity");
                var intent = new AndroidJavaObject("android.content.Intent",
                    "android.settings.APPLICATION_DETAILS_SETTINGS");

                using (var uriClass = new AndroidJavaClass("android.net.Uri"))
                {
                    var uri = uriClass.CallStatic<AndroidJavaObject>("parse",
                        "package:" + activity.Call<string>("getPackageName"));
                    intent.Call<AndroidJavaObject>("setData", uri);
                }

                activity.Call("startActivity", intent);
            }
            catch (Exception e)
            {
                AddLog("Failed to open app settings: " + e.Message);
            }
        }

        private static string LogGlassStatus(GlassStatus status)
        {
            var log = "";
            if (status == null) return log;
            log += $"\nBluetooth = {status.BluetoothStatus}";
            log += $"\nWi-Fi = {status.WifiStatus}";
            log += $"\nSpacesAvailable = {status.SpacesAvailable}";
            return log;
        }
        
        private static string LogSwitchMicrophoneStatus(SwitchMicrophoneStatus status)
        {
            var log = "";
            if (status == null) return log;
            log += $"\nwearingMicMode = {status.WearingMicMode}";
            log += $"\nfrontMicMode = {status.FrontMicMode}";
            log += $"\nmixMode = {status.MixMode}";
            return log;
        }

        // 確認用ログテキストを追加
        private void AddLog(string text, int outputIntervalTimeMillis = 0 )
        {
            Debug.LogWarning(text);

            var currentTime = DateTime.Now;
            var elapsedMilliseconds = (currentTime - _lastLogUpdateTime).TotalMilliseconds;

            if (elapsedMilliseconds >= outputIntervalTimeMillis)
                _mainThreadContext.Post(_ =>
                {
                    logText.text = CreateLogTime() + "\n" + text + "\n\n" + logText.text;

                    // スクロールポジションを一番上へ移動
                    scrollRect.verticalNormalizedPosition = 1.0f;
                }, null);

            _lastLogUpdateTime = currentTime;
        }

        // 確認用ログの時間
        private static string CreateLogTime()
        {
            var now = DateTime.Now;
            return now.ToString("HH:mm:ss.") + now.Millisecond;
        }

        private void DisplayPreview(SpacesModeStatus status)
        {
            if (preview != null) preview.SetActive(status != SpacesModeStatus.On);
        }
    }
}