using UnityEngine;
using UnityEngine.UI;
using System.Collections.Generic;
using System;
using System.Collections;

namespace com.nttqonoq.devices.android.mirzalibrary.samples
{
    /// <summary>
    /// 選択肢ダイアログ
    /// </summary>
    public class OptionListDialog : MonoBehaviour
    {
        [Header("UI References")] [SerializeField]
        private Text titleText;

        [SerializeField] private RectTransform contentContainer;
        [SerializeField] private Button okButton;
        [SerializeField] private Button cancelButton;
        [SerializeField] private GameObject dropdownPrefab;

        [Header("Layout Settings")] 
        [SerializeField] private float spacing = 10f;
        [SerializeField] private float itemHeight = 30f;

        private List<(string label, List<object> options)> _items;
        private readonly List<Dropdown> _dropdowns = new List<Dropdown>();
        private readonly  List<List<object>> _originalValues = new List<List<object>>();
        private Action<List<object>> _onSelectionComplete;

        public static OptionListDialog Show(
            string title,
            List<(string label, List<object> options)> items,
            Action<List<object>> callback,
            GameObject prefab,
            Transform parent = null)
        {
            var instance = Instantiate(prefab, parent);
            var dialog = instance.GetComponent<OptionListDialog>();
            if (dialog == null)
            {
                Debug.LogError("Prefab must have OptionListDialog component attached!");
                Destroy(instance);
                return null;
            }

            dialog._items = items;
            dialog.Initialize(title, callback);
            return dialog;
        }

        private void Initialize(string title, System.Action<List<object>> callback)
        {
            titleText.text = title;
            _onSelectionComplete = callback;
            StartCoroutine(CreateDropdowns());
        }

        private IEnumerator CreateDropdowns()
        {
            // Wait for layout to be rebuilt
            LayoutRebuilder.ForceRebuildLayoutImmediate(contentContainer);
            yield return null;

            CreateDropdowns(_items);
            SetupButtons();
        }

        private void CreateDropdowns(List<(string label, List<object> options)> items)
        {
            ClearExistingDropdowns();

            LayoutRebuilder.ForceRebuildLayoutImmediate(contentContainer);
            var containerRect = contentContainer.GetComponent<RectTransform>();
            var optionsWidth = containerRect.rect.width;

            if (optionsWidth <= 0)
            {
                Debug.LogWarning("contentContainer width is 0 or negative. Using fallback values.");
                optionsWidth = 400f; // 横幅が取れない場合は固定
            }

            var labelWidth = optionsWidth; 
            var dropdownWidth = optionsWidth;
            var currentY = 0f;

            foreach (var (label, options) in items)
            {
                // 選択肢タイトル
                var labelObj = new GameObject($"Label_{label}", typeof(RectTransform), typeof(Text));
                var labelRect = labelObj.GetComponent<RectTransform>();
                var labelText = labelObj.GetComponent<Text>();
                SetupLabel(labelObj, labelText, labelWidth, labelRect, label, currentY);

                // 選択項目リスト
                var dropdownObj = Instantiate(dropdownPrefab, contentContainer);
                var dropdownRect = dropdownObj.GetComponent<RectTransform>();
                var dropdown = dropdownObj.GetComponent<Dropdown>();
                var optionStrings = new List<string>();
                foreach (var option in options)
                {
                    optionStrings.Add(option.ToString());
                }
                SetupDropdown(dropdownObj, dropdownRect, dropdownWidth, dropdown, optionStrings, labelWidth, currentY);
                _dropdowns.Add(dropdown);
                _originalValues.Add(options); // 元のオプションを保存しておく

                currentY -= (itemHeight + spacing);
            }

            // 高さ調整
            contentContainer.sizeDelta = new Vector2(
                contentContainer.sizeDelta.x,
                Mathf.Abs(currentY) + spacing
            );
        }
        
        // 選択肢タイトル
        private void SetupLabel(
            GameObject labelObj,
            Text labelText,
            float labelWidth,
            RectTransform labelRect,
            string labelContent,
            float yPosition)
        {
            labelObj.transform.SetParent(contentContainer, false);

            labelRect.anchorMin = new Vector2(0, 1);
            labelRect.anchorMax = new Vector2(0, 1);
            labelRect.pivot = new Vector2(0, 1);
            labelRect.sizeDelta = new Vector2(labelWidth, itemHeight);
            labelRect.anchoredPosition = new Vector2(0, yPosition);

            labelText.text = labelContent;
            labelText.alignment = TextAnchor.MiddleLeft;
            labelText.color = Color.black;
            labelText.font = Resources.GetBuiltinResource<Font>("Arial.ttf");
            labelText.fontSize = 48;
        }

        // 選択項目リスト
        private void SetupDropdown(
            GameObject dropdownObj,
            RectTransform dropdownRect,
            float dropdownWidth,
            Dropdown dropdown,
            List<string> options,
            float labelWidth,
            float yPosition)
        {
            dropdownRect.anchorMin = new Vector2(0, 1);
            dropdownRect.anchorMax = new Vector2(0, 1);
            dropdownRect.pivot = new Vector2(0, 1);
            dropdownRect.sizeDelta = new Vector2(dropdownWidth, itemHeight);
            dropdownRect.anchoredPosition = new Vector2(labelWidth + spacing, yPosition);

            dropdown.ClearOptions();
            dropdown.AddOptions(options);
        }

        // フッターボタン
        private void SetupButtons()
        {
            okButton.onClick.RemoveAllListeners();
            cancelButton.onClick.RemoveAllListeners();

            okButton.onClick.AddListener(OnOkClicked);
            cancelButton.onClick.AddListener(OnCancelClicked);
        }

        private void OnOkClicked()
        {
            var selectedItems = new List<object>();
            for (int i = 0; i < _dropdowns.Count; i++)
            {
                int selectedIndex = _dropdowns[i].value;
                selectedItems.Add(_originalValues[i][selectedIndex]);
            }

            _onSelectionComplete?.Invoke(selectedItems);
            Close();
        }

        private void OnCancelClicked()
        {
            Close();
        }

        private void Close()
        {
            Destroy(gameObject);
        }

        private void ClearExistingDropdowns()
        {
            foreach (Transform child in contentContainer)
            {
                Destroy(child.gameObject);
            }

            _dropdowns.Clear();
            _originalValues.Clear();
        }

        private void OnDestroy()
        {
            if (okButton != null) okButton.onClick.RemoveAllListeners();
            if (cancelButton != null) cancelButton.onClick.RemoveAllListeners();
        }
    }
}
