using System;
using UnityEngine;

namespace com.nttqonoq.devices.android.mirzalibrary
{
    /// <summary>
    /// 結果の状態 Enum
    /// </summary>
    public enum State
    {
        Success = 0, // 成功
        ServiceNotConnected = 1, // サービス未接続
        ServiceError = 2, // サービス接続エラー
        VersionInconsistent = 3, // バージョン不整合
        GlassNotPaired = 4, // グラスデバイス未ペアリング
        GlassNotConnected = 5, // グラスデバイス未接続
        GlassNotConnectedOrTimeout = 6, // グラスデバイス未接続またはタイムアウト
        UnexpectedResponseData = 7, // 想定外のレスポンスデータ
        NotFoundMirzaApp = 8, // MiRZAアプリが見つからない
        MirzaAppNotInSpacesMode = 9, // MiRZAアプリはSpacesモードではない
        Unknown = 10 // 不明
    }

    /// <summary>
    /// 連携結果
    /// </summary>
    /// <typeparam name="T">関連データ型</typeparam>
    public class Result<T>
    {
        public Result(State state, T data, string errorMessage)
        {
            State = state;
            Data = data;
            ErrorMessage = errorMessage;
        }

        public State State { get; }

        public T Data { get; }

        public string ErrorMessage { get; }

        public static Result<T> Convert(AndroidJavaObject resultObj)
        {
            try
            {
                var state = (State)resultObj.Call<AndroidJavaObject>("getState").Call<int>("getCode");
                var data = resultObj.Call<AndroidJavaObject>("getData");
                var errorMessage = resultObj.Call<AndroidJavaObject>("getErrorMessage");

                if (state != State.Success) return new Result<T>(state, default, null);
                
                if (data != null)
                {
                    T convertedData;
                    if (typeof(T) == typeof(string))
                        convertedData = (T)(object)data?.Call<string>("toString");
                    else if (typeof(T) == typeof(int))
                        convertedData = (T)(object)data?.Call<int>("intValue");
                    else if (typeof(T) == typeof(bool))
                        convertedData = (T)(object)data?.Call<bool>("booleanValue");
                    else if (typeof(T) == typeof(float))
                        convertedData = (T)(object)data.Call<float>("floatValue");
                    else if (typeof(T) == typeof(double))
                        convertedData = (T)(object)data?.Call<double>("doubleValue");
                    else if (typeof(T) == typeof(GlassStatus))
                        convertedData = (T)(object)GlassStatus.Convert(data);
                    else if (typeof(T) == typeof(ChargeStatus))
                        convertedData = (T)(object)(ChargeStatus)data.Call<int>("getCode");
                    else if (typeof(T) == typeof(SpacesModeStatus))
                        convertedData = (T)(object)(SpacesModeStatus)data.Call<int>("getCode");
                    else if (typeof(T) == typeof(SwitchMicrophoneStatus))
                        convertedData = (T)(object)SwitchMicrophoneStatus.Convert(data);                    
                    else
                        throw new InvalidOperationException($"Unsupported type: {typeof(T)}");

                    return new Result<T>(state, convertedData, null);
                }

                if (errorMessage != null)
                {
                    return new Result<T>(state, default, errorMessage.Call<string>("toString"));
                }
                
                return new Result<T>(state, default, null);    
                
            }
            catch (NullReferenceException ex)
            {
                Debug.Log(ex);
                return new Result<T>(State.UnexpectedResponseData, default, null);
            }
            catch (InvalidOperationException ex)
            {
                Debug.Log(ex);
                return new Result<T>(State.Unknown, default, null);
            }
        }
    }
}