﻿using System.Threading;
using UnityEngine;
using UnityEngine.Events;
using TouchType = com.nttqonoq.devices.android.mirzalibrary.TouchType;
using com.nttqonoq.devices.android.mirzalibrary;
using NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Runtime;

namespace NTTQONOQ.Android.MiRZA.Library.Samples.TouchSensorSample
{
    // グラスのタッチセンサーの状態に応じて、登録したイベントを実行する
    public class GlassesTouchSensorEventTrigger : MonoBehaviour
    {
        public TouchType TouchType;
        public TouchOperation TouchOperation;
        public int XCoordinate; //タッチしている箇所の横方向の座標：0-255
        public int YCoordinate; //タッチしている箇所の縦方向の座標：0のみ
        public int Movement; //加速度

        [SerializeField] private bool debugLogEnable = false;
        [SerializeField] private UnityEvent singleTapEvents;
        [SerializeField] private UnityEvent doubleTapEvents;
        [SerializeField] private UnityEvent tripleTapEvents;
        [SerializeField] private UnityEvent LongPressInitial;
        [SerializeField] private UnityEvent LongPressRepeat;
        [SerializeField] private UnityEvent moveEvents;
        [SerializeField] private UnityEvent previousEvents;
        [SerializeField] private UnityEvent nextEvents;
        [SerializeField] private UnityEvent upEvents;
        [SerializeField] private UnityEvent downEvents;

        private SynchronizationContext _mainThreadContext;
        private IMirzaPlugin _mirzaPlugin;
        [SerializeField] private bool simulateEnable = true;// Unity Editor状でタッチセンサーをシミュレートするかどうか。操作方法は矢印キー。
        private float elapsedTimeSinceTap = 0f;
        private float elapsedTimeSinceLongTap = 0f;
        private float judgmentTimeForContinuousTap = 0.5f;
        private float longPressRepeatIntervalTime = 0.3f;
        private float judgmentTimeForLongTap = 1f;
        private bool isLongTap = false;
        private int tapCount = 0;

        private void Start()
        {
            _mainThreadContext = SynchronizationContext.Current;

            // ライブラリの初期化
            _mirzaPlugin = MirzaPluginProvider.MirzaPlugin;
            _mirzaPlugin?.SetLogEnable(true);

            StartLibrary();
        }

        private void StartLibrary()
        {
            if (_mirzaPlugin == null) return;

            _mirzaPlugin.OnGlassTouchGestureStatusChanged += HandleGlassTouchGestureStatusChanged;
            _mirzaPlugin.StartMonitoring();
        }

        private void Update()
        {
            if (simulateEnable)
            {
                SimulateWithKeyboard();
            }
            //Debug.Log("[TouchSensor] XCoordinate:" + XCoordinate + ", YCoordinate:" + YCoordinate);
        }

        // Unity Editor用のシミュレーション
        // タッチセンサーのイベントをPCのキー操作でシミュレートする（MiRZA Library v1.1.0）
        // ←キー：後方向へスワイプ（Next）
        // →キー：前方向へスワイプ（Previous）
        // ↓キーを1回：シングルタップ（Single）
        // ↓キーを2回：ダブルタップ（Double）
        // ↓キーを3回：トリプルタップ（Triple）
        // ↓キーを長押し：ロングタップ（LongPressInitial、LongPressRepeat）
        private void SimulateWithKeyboard()
        {
            // スワイプ
            if (Input.GetKeyDown(KeyCode.LeftArrow))// 後方向へスワイプ（Next）
            {
                if (debugLogEnable)
                {
                    Debug.Log("TouchSensor: Down");
                    Debug.Log("TouchSensor: Move");
                    Debug.Log("TouchSensor: Up");
                    Debug.Log("TouchSensor: Next");
                }
                downEvents.Invoke();
                moveEvents.Invoke();
                upEvents.Invoke();
                nextEvents.Invoke();

                // ロングタップのキャンセル
                elapsedTimeSinceLongTap = 0f;
                isLongTap = false;
            }
            if (Input.GetKeyDown(KeyCode.RightArrow))// 前方向へスワイプ（Previous）
            {
                if (debugLogEnable)
                {
                    Debug.Log("TouchSensor: Down");
                    Debug.Log("TouchSensor: Move");
                    Debug.Log("TouchSensor: Up");
                    Debug.Log("TouchSensor: Previous");
                }
                downEvents.Invoke();
                moveEvents.Invoke();
                upEvents.Invoke();
                previousEvents.Invoke();

                // ロングタップのキャンセル
                elapsedTimeSinceLongTap = 0f;
                isLongTap = false;
            }

            // ロングタップの判定
            if (isLongTap == false && Input.GetKey(KeyCode.DownArrow))
            {
                if (elapsedTimeSinceLongTap > judgmentTimeForContinuousTap - 0.05f)// 長押ししたら、シングルタップのキャンセル
                {
                    tapCount = 0;
                    elapsedTimeSinceTap = 0f;
                }
                if (elapsedTimeSinceLongTap > judgmentTimeForLongTap) //一定時間が経過したら、ロングタップ判定
                {
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: LongPressInitial");
                    }
                    LongPressInitial.Invoke();
                    //elapsedTimeSinceLongTap = 0f;
                    isLongTap = true;
                }
                elapsedTimeSinceLongTap += Time.deltaTime;
            }
            if (isLongTap && Input.GetKey(KeyCode.DownArrow))
            {
                if (elapsedTimeSinceLongTap >= longPressRepeatIntervalTime)
                {
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: LongPressRepeat");
                    }
                    LongPressRepeat.Invoke();
                    elapsedTimeSinceLongTap = 0.0f;
                }
                elapsedTimeSinceLongTap += Time.deltaTime;
            }
            if (Input.GetKeyUp(KeyCode.DownArrow))// 指を離したらリセット
            {
                upEvents.Invoke();
                if (debugLogEnable)
                {
                    Debug.Log("TouchSensor: Up");
                }
                elapsedTimeSinceLongTap = 0f;
                isLongTap = false;
            }

            // タップの判定
            if (Input.GetKeyDown(KeyCode.DownArrow))
            {
                downEvents.Invoke();
                if (debugLogEnable)
                {
                    Debug.Log("TouchSensor: Down");
                }
                tapCount++;
            }
            if (tapCount >= 1)
            {
                if (elapsedTimeSinceTap > judgmentTimeForContinuousTap) //一定時間が経過したら、タップの種類を判定
                {
                    switch (tapCount)
                    {
                        case 1:
                            if (debugLogEnable)
                            {
                                Debug.Log("TouchSensor: SingleTap");
                            }
                            singleTapEvents.Invoke();
                            tapCount = 0;
                            elapsedTimeSinceTap = 0;
                            break;
                        case 2:
                            if (debugLogEnable)
                            {
                                Debug.Log("TouchSensor: DoubleTap");
                            }
                            doubleTapEvents.Invoke();
                            tapCount = 0;
                            elapsedTimeSinceTap = 0;
                            break;
                        case 3:
                            if (debugLogEnable)
                            {
                                Debug.Log("TouchSensor: TripleTap");
                            }
                            tripleTapEvents.Invoke();
                            tapCount = 0;
                            elapsedTimeSinceTap = 0;
                            break;
                    }
                }
                elapsedTimeSinceTap += Time.deltaTime;
            }
        }

        // グラスジェスチャーの変化通知イベントハンドラー
        private void HandleGlassTouchGestureStatusChanged(GlassTouchGestureStatus status)
        {
            // センサー情報の更新
            TouchType = status.Type;
            TouchOperation = status.Operation;
            XCoordinate = status.XCoordinate;
            YCoordinate = status.YCoordinate;
            Movement = status.Movement;

            // グラスのタッチセンサーの状態に応じて、登録したイベントを実行する

            // シングルタップ
            if (status.Operation == TouchOperation.Single)
            {
                _mainThreadContext.Post((_) =>
                {
                    singleTapEvents.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: Single");
                    }
                }, null);
            }

            // ダブルタップ
            if (status.Operation == TouchOperation.Double)
            {
                _mainThreadContext.Post((_) =>
                {
                    doubleTapEvents.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: Double");
                    }
                }, null);
            }

            // トリプルタップ
            if (status.Operation == TouchOperation.Triple)
            {
                _mainThreadContext.Post((_) =>
                {
                    tripleTapEvents.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: Triple");
                    }
                }, null);
            }

            // ロングタップ(初回のみ)
            if (status.Operation == TouchOperation.LongPressInitial)
            {
                _mainThreadContext.Post((_) =>
                {
                    LongPressInitial.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: LongPressInitial");
                    }
                }, null);
            }

            // ロングタップ(初回以降)
            if (status.Operation == TouchOperation.LongPressRepeat)
            {
                _mainThreadContext.Post((_) =>
                {
                    LongPressRepeat.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: LongPressRepeat");
                    }
                }, null);
            }

            // 移動
            if (status.Operation == TouchOperation.Move)
            {
                _mainThreadContext.Post((_) =>
                {
                    moveEvents.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: Move");
                    }
                }, null);
            }

            // 前
            if (status.Operation == TouchOperation.Previous)
            {
                _mainThreadContext.Post((_) =>
                {
                    previousEvents.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: Previous");
                    }
                }, null);
            }

            // 後
            if (status.Operation == TouchOperation.Next)
            {
                _mainThreadContext.Post((_) =>
                {
                    nextEvents.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: Next");
                    }
                }, null);
            }

            // アップ
            if (status.Operation == TouchOperation.Up)
            {
                _mainThreadContext.Post((_) =>
                {
                    upEvents.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: Up");
                    }
                }, null);
            }

            // ダウン
            if (status.Operation == TouchOperation.Down)
            {
                _mainThreadContext.Post((_) =>
                {
                    downEvents.Invoke();
                    if (debugLogEnable)
                    {
                        Debug.Log("TouchSensor: Down");
                    }
                }, null);
            }
        }
    }
}
