using System;
using System.Collections;
using Unity.Collections.LowLevel.Unsafe;
using UnityEngine;
using UnityEngine.UI;
using UnityEngine.XR.ARFoundation;
using UnityEngine.XR.ARSubsystems;

namespace NTTQONOQ.Android.MiRZA.Library.Samples.GPTImageAnalyzerSample
{
    public unsafe class ImageCapture: MonoBehaviour
    {
        [SerializeField] private RawImage _rawImage;

        [SerializeField] private float _startDelay = 3.0f;

        private ARCameraManager _cameraManager;

        private Texture2D _cameraTexture;

        private Vector2 _initialTextureSize;

        private float _initialAspectRatio;

        private IEnumerator Start()
        {
            yield return new WaitForSeconds(_startDelay);

            Debug.Log("Start ImageCapture");
            _cameraManager = FindFirstObjectByType<ARCameraManager>(FindObjectsInactive.Include);

            if(_cameraManager == null)
            {
                Debug.LogError("ARCameraManager not found in the scene.");
                yield break;
            }

            if (_cameraManager.enabled == false)
            {
                Debug.Log("Enabling ARCameraManager");
                _cameraManager.enabled = true;
            }

            _cameraManager.frameReceived += OnFrameReceived;
            _initialTextureSize = _rawImage.rectTransform.sizeDelta;
            _initialAspectRatio = _initialTextureSize.x / _initialTextureSize.y;
        }

        private void OnFrameReceived(ARCameraFrameEventArgs args)
        {
            // Ensure the camera is ready to provide images
            if (!_cameraManager.TryAcquireLatestCpuImage(out var image))
            {
                return;
            }

            try
            {
                var format = TextureFormat.RGBA32;
                var dimensions = image.dimensions;

                // Check if the texture needs to be created or resized
                if (_cameraTexture == null ||
                    _cameraTexture.width != dimensions.x ||
                    _cameraTexture.height != dimensions.y)
                {
                    Debug.Log("Creating camera texture: " + dimensions.x + "x" + dimensions.y);
                    _cameraTexture = new Texture2D(dimensions.x, dimensions.y, format, false);
                    // Resize the RawImage to fit the new texture size
                    var outputAspectRatio = dimensions.x / (float)dimensions.y;
                    if (outputAspectRatio > _initialAspectRatio)
                    {
                        _rawImage.rectTransform.sizeDelta =
                            new Vector2(_initialTextureSize.x, _initialTextureSize.x / outputAspectRatio);
                    }
                    else
                    {
                        _rawImage.rectTransform.sizeDelta =
                            new Vector2(_initialTextureSize.y * outputAspectRatio, _initialTextureSize.y);
                    }
                }

                // Convert the image to the texture
                var rawTextureData = _cameraTexture.GetRawTextureData<byte>();
                var rawTexturePtr = new IntPtr(rawTextureData.GetUnsafePtr());
                var conversionParams = new XRCpuImage.ConversionParams(image, format)
                {
                    inputRect = new RectInt(0, 0, image.width, image.height),
                    outputDimensions = dimensions
                };
                image.Convert(conversionParams, rawTexturePtr, rawTextureData.Length);
                _cameraTexture.Apply();
                _rawImage.texture = _cameraTexture;
            }
            catch (Exception e)
            {
                Debug.LogException(e);
            }
            finally
            {
                image.Dispose();
            }
        }

        private void OnDestroy()
        {
            if (_cameraManager != null)
            {
                _cameraManager.frameReceived -= OnFrameReceived;
            }
            if (_cameraTexture != null)
            {
                Destroy(_cameraTexture);
                _cameraTexture = null;
            }
        }
    }
}