using System;
using System.Collections;
using System.Collections.Generic;
using Newtonsoft.Json;
using UnityEngine;
using UnityEngine.UI;
using UnityEngine.Networking;
using TMPro;
using UnityEngine.Scripting;

namespace NTTQONOQ.Android.MiRZA.Library.Samples.GPTImageAnalyzerSample
{
    public class GPTImageAnalyzer : MonoBehaviour
    {
        [Serializable]
        private class OpenAIResponse
        {
            public Choice[] choices;

            [Serializable]
            public class Choice
            {
                public Message message;
            }

            [Serializable]
            public class Message
            {
                public string content;
            }
        }

        private static readonly string OpenaiApiUrl = "https://api.openai.com/v1/chat/completions";

        [Preserve]
        internal sealed class MarkerForInternet : UnityWebRequest { }

        // 「openAI_APIKey_Text」を設定済みの場合は必要無し(GPT-4oを想定)
        [SerializeField] private string _openAiApiKey;

        // 「openAI_APIKey」を設定済みの場合は必要無し(GPT-4oを想定)
        [SerializeField] private TextAsset _openAiApiKeyText;

        [SerializeField, TextArea(3, 10)] private string _userPrompt = "この画像には何が写っていますか？";

        // 解析の状況
        [SerializeField] private TextMeshProUGUI _descriptionStateText;

        // 解析結果の説明
        [SerializeField] private TextMeshProUGUI _descriptionText;

        // GPT-4oで出力されたテキストを音声に変換する
        [SerializeField] private GoogleTextToSpeechConverter _googleTextToSpeechConverter;

        // 解析に使用する画像
        [SerializeField] private RawImage _targetImage;

        private void Awake()
        {
            // APIキーのテキストファイルが設定済みの場合は、そのAPIキーを使用する
            if (_openAiApiKeyText)
            {
                _openAiApiKey = _openAiApiKeyText.text;
            }

            if (string.IsNullOrEmpty(_openAiApiKey))
            {
                Debug.LogError("OpenAI API Key is not set. Please set it.");
            }
        }

        public void Analyze()
        {
            StartCoroutine(AnalyzeCoroutine());
        }

        private IEnumerator AnalyzeCoroutine()
        {
            if (_targetImage == null)
            {
                _descriptionStateText.text = "No image to analyze.";
                yield break;
            }

            var tex = _targetImage.texture;
            var sw = tex.width;
            var sh = tex.height;
            var result = new Texture2D(sw, sh, TextureFormat.RGBA32, false);
            var currentRT = RenderTexture.active;
            var rt = new RenderTexture(sw, sh, 32);

            Graphics.Blit(tex, rt);
            RenderTexture.active = rt;

            result.ReadPixels(new Rect(0, 0, rt.width, rt.height), 0, 0);
            result.Apply();
            RenderTexture.active = currentRT;

            var bytes = result.EncodeToPNG();
            var base64Image = Convert.ToBase64String(bytes);

            var requestBody = new
            {
                model = "gpt-4o",
                messages = new[]
                {
                new
                {
                    role = "user",
                    content = new List<object>
                    {
                        new { type = "text", text = _userPrompt },
                        new { type = "image_url", image_url = new { url = $"data:image/jpeg;base64,{base64Image}" } }
                    }
                }
            },
                max_tokens = 300
            };

            // JsonUtility does not support anonymous types, so we use Newtonsoft.Json for serialization.
            var json = JsonConvert.SerializeObject(requestBody);

            using var www = UnityWebRequest.PostWwwForm(OpenaiApiUrl, " ");
            www.uploadHandler = new UploadHandlerRaw(System.Text.Encoding.UTF8.GetBytes(json));
            www.uploadHandler.contentType = "application/json";
            www.SetRequestHeader("Authorization", $"Bearer {_openAiApiKey}");
            www.SetRequestHeader("Content-Type", "application/json");

            _descriptionStateText.text = "Analysis in progress...";
            // googleTextToSpeechConverterが設定済みの場合は、音声出力を行う
            if (_googleTextToSpeechConverter)
            {
                _googleTextToSpeechConverter.SynthesizeAndPlay("画像を解析中です");
            }

            yield return www.SendWebRequest();

            if (www.result != UnityWebRequest.Result.Success)
            {
                _descriptionStateText.text = "An error occurred: " + www.error;
            }
            else
            {
                var response = JsonConvert.DeserializeObject<OpenAIResponse>(www.downloadHandler.text);
                var description = response.choices[0].message.content;

                _descriptionText.text = description;
                // googleTextToSpeechConverterが設定済みの場合は、音声出力を行う
                if (_googleTextToSpeechConverter)
                {
                    _googleTextToSpeechConverter.SynthesizeAndPlay(description);
                }
                _descriptionStateText.text = "Analysis complete";
            }
        }
    }
}