using System.Collections;
using Qualcomm.Snapdragon.Spaces;
using Qualcomm.Snapdragon.Spaces.Samples;
using UnityEngine;
using UnityEngine.EventSystems;

namespace NTTQONOQ.Android.MiRZA.Controller
{
    public class TouchpadRayDragHandler : MonoBehaviour, IPointerDownHandler, IPointerUpHandler, IBeginDragHandler, IDragHandler, IEndDragHandler
    {
        public Camera xrCamera;
        public XRCameraPoseFollower cameraPoseFollower;
        public CanvasControllerCompanion inputDevice;
        public float touchpadRaySensitivity = 1.5f;

        
        private float _lastTapTime = 0f;
        private int _tapCount = 0;
        private Vector2 _prevFingerPosition = new Vector2(0, 0);
        private const float MultiTapThreshold = 0.3f; // タップを判定する間隔
        private bool _isMultiTapCoroutineRunning = false;
        private bool _pointerUp = true;

        private void Awake()
        {
            if (xrCamera == null)
            {
                xrCamera = OriginLocationUtility.GetOriginCamera(true);
            }
        }

        public void OnBeginDrag(PointerEventData eventData)
        {
            // ダブルタップの場合
            if (_tapCount == 2)
            {
                inputDevice.SendTouchpadEvent(1, NormalizedPosition(eventData.position));
            }
            
            _prevFingerPosition = NormalizedPosition(eventData.position);
        }

        public void OnDrag(PointerEventData eventData)
        {
            // ダブルタップの場合
            if (_tapCount == 2)
            {
                inputDevice.SendTouchpadEvent(2, NormalizedPosition(eventData.position));
            }
            
            // 変位量計算
            Vector2 position = NormalizedPosition(eventData.position);
            Vector2 displacementVec2 = position - _prevFingerPosition;
            displacementVec2 *= touchpadRaySensitivity;
            _prevFingerPosition = position;

            //Debug.Log($"{eventData.position} - {position} - {_prevFingerPosition} - {displacementVec2}");

            // レイの回転を更新
            Vector3 rayDirection = cameraPoseFollower.transform.forward;
            Quaternion pointerQuaternion = CalculatePointerQuaternion(rayDirection, displacementVec2);
            cameraPoseFollower.transform.rotation = pointerQuaternion;
        }

        public void OnEndDrag(PointerEventData eventData)
        {
            inputDevice.SendTouchpadEvent(0, NormalizedPosition(eventData.position));
        }

        public void OnPointerDown(PointerEventData eventData)
        {
            //inputDevice.SendTouchpadEvent(1, NormalizedPosition(eventData.position));

            _pointerUp = false;

            // 現在の時間を取得
            float currentTime = Time.time;

            // 前回のタップからの経過時間を計算
            if (currentTime - _lastTapTime <= MultiTapThreshold)
            {
                _tapCount++;
            }
            else
            {
                _tapCount = 1;
                _lastTapTime = currentTime;     // 最後のタップ時間を更新
            }

            if (!_isMultiTapCoroutineRunning)
            {
                StartCoroutine(SingleTapCoroutine(eventData));
            }
        }

        public void OnPointerUp(PointerEventData eventData)
        {
            //inputDevice.SendTouchpadEvent(0, NormalizedPosition(eventData.position));

            _pointerUp = true;

            if (!_isMultiTapCoroutineRunning)
            {
                inputDevice.SendTouchpadEvent(0, NormalizedPosition(eventData.position));
            }
        }

        private IEnumerator SingleTapCoroutine(PointerEventData eventData)
        {
            _isMultiTapCoroutineRunning = true;
            
            // 現在の時間を取得
            float currentTime = Time.time;

            while (currentTime - _lastTapTime <= MultiTapThreshold)
            {
                currentTime = Time.time;        // 現在の時間を更新

                yield return new WaitForEndOfFrame();
            }

            // タップ判定
            // シングルタップかつPointerUpの場合
            // またはダブルタップ以上の場合
            if ((_tapCount == 1 && _pointerUp) || _tapCount >= 2)
            {
                // 選択開始を送信
                inputDevice.SendTouchpadEvent(1, NormalizedPosition(eventData.position));
            }

            yield return new WaitForEndOfFrame();

            // pointerUpがtrueの場合、選択解除を送信
            if (_pointerUp)
            {
                inputDevice.SendTouchpadEvent(0, NormalizedPosition(eventData.position));
                _tapCount = 0; // カウントをリセット
            }
            
            _isMultiTapCoroutineRunning = false;
        }
        
        public void ResetPointer()
        {
            Vector3 forward = xrCamera.transform.forward;
            forward.y = 0;
            cameraPoseFollower.transform.forward = forward; // rotation = Quaternion.Euler(0, 0, 0);
        }


        private Vector2 NormalizedPosition(Vector2 eventPosition)
        {
            RectTransform rectTransform = GetComponent<RectTransform>();
            float halfWidth = rectTransform.rect.width / 2;
            float halfHeight = rectTransform.rect.height / 2;

            if (RectTransformUtility.ScreenPointToLocalPointInRectangle(
                    GetComponent<RectTransform>(),
                    eventPosition,
                    null,
                    out var localizedPosition))
            {
                Vector2 normalized = new Vector2(localizedPosition.x / halfWidth, localizedPosition.y / halfHeight);
                return normalized;
            }

            return Vector2.zero;
        }
        
        /// <summary>
        /// 現在のベクトル と2Dの変位量を基に、新しい方向を表すクォータニオンを計算
        /// </summary>
        /// <param name="nowVector"></param>
        /// <param name="displacement2D"></param>
        /// <returns></returns>
        private Quaternion CalculatePointerQuaternion(Vector3 nowVector, Vector2 displacement2D)
        {
            // Y軸を基準とする(upをY軸とする)
            Vector3 axisY = Vector3.up;

            // Y軸と現在のベクトルでX軸を計算
            Vector3 axisX = -Vector3.Cross(nowVector, axisY);

            // 変位ベクトルを計算
            Vector3 displacement = (axisX * displacement2D.x + axisY * displacement2D.y);

            // 変位ベクトルと現在のベクトルで新しいベクトルを計算
            Vector3 newVec = (nowVector + displacement);

            // 新しいベクトルへの回転を計算
            Quaternion newRayQua = Quaternion.FromToRotation(Vector3.forward, newVec);

            // 計算したQuaternionを返す
            return newRayQua;
        }
    }
}
