﻿#if USING_SNAPDRAGON_SPACES_SDK && USING_MIRZA_SDK
using System;
using System.Collections;
using UnityEngine;
using UnityEngine.SceneManagement;
using com.nttqonoq.devices.android.mirzalibrary;
using Qualcomm.Snapdragon.Spaces;
using static Qualcomm.Snapdragon.Spaces.SpacesGlassStatus;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Runtime
{
    // MiRZAアプリを立ち上げずに、グラスをMRモードに自動で切り替える（DRF専用）
    public class MrModeSwitcher : MonoBehaviour
    {
        private Action<ConnectionState> OnConnectionStateChanged;

        private IMirzaPlugin _mirzaPlugin;
        private DynamicOpenXRLoader _dynamicOpenXRLoader;
        private SpacesGlassStatus _spacesGlassStatus;

        private bool _isMiRZAServiceConnected = false;
        private bool _isMRMode = false;
        private ConnectionState _glassConnectionState = ConnectionState.Disconnected;

        [Tooltip("If true, the scene specified by sceneName will be loaded automatically after switching to MR Mode.")]
        [SerializeField]
        private bool changeScene = false;

        [Tooltip("Name of the scene that will be loaded automatically after switching to MR Mode.")]
        [SerializeField]
        private string sceneName = "Main Scene";

        private void Start()
        {
            _spacesGlassStatus = SpacesGlassStatus.Instance;
            OnConnectionStateChanged += ConnectionStateChanged;
            _dynamicOpenXRLoader = FindFirstObjectByType<DynamicOpenXRLoader>();

            _mirzaPlugin = MirzaPluginProvider.MirzaPlugin;

            _mirzaPlugin?.SetLogEnable(true);

            StartCoroutine(CheckGlassStatus());

            StartLibrary();
        }

        private void ConnectionStateChanged(ConnectionState state)
        {
            Debug.Log("[MrModeSwitcher] ConnectionStateChanged " + state.ToString());
            if (state == ConnectionState.Connected)
            {
                OnOpenXRAvailable();
            }
            else
            {
                OnOpenXRNotAvailable();
            }
        }

        private void OnDestroy()
        {
            StopLibrary();
        }

        private void OnApplicationPause(bool pauseStatus)
        {
            if (pauseStatus)
            {
                StopLibrary();
            }
            else
            {
                StartLibrary();
            }
        }

        private void StartLibrary()
        {
            if (_mirzaPlugin == null) return;
            _mirzaPlugin.OnServiceStateChanged += HandleServiceStateChanged;
            _mirzaPlugin.StartMonitoring();
        }

        private void StopLibrary()
        {
            if (_mirzaPlugin == null) return;
            _mirzaPlugin.OnServiceStateChanged -= HandleServiceStateChanged;
        }

        private void HandleServiceStateChanged(ServiceState state)
        {
            Debug.Log($"[MrModeSwitcher] Service State : {state}");
            if (state == ServiceState.Connected)
            {
                try
                {
                    MRModeSequence();
                }
                catch (Exception ex)
                {
                    Debug.LogError(ex.ToString());
                }
                _isMiRZAServiceConnected = true;
            }
            else if (state == ServiceState.Disconnected)
            {
                _isMiRZAServiceConnected = false;
            }
        }

        private void MRModeSequence()
        {
            Debug.Log("[MrModeSwitcher] MRModeSequence start");
            if (_mirzaPlugin == null)
            {
                Debug.LogWarning("[MrModeSwitcher] _mirzaPlugin is null ?");
                _mirzaPlugin = MirzaPluginProvider.MirzaPlugin;
            }
            var status = _mirzaPlugin.GetGlassStatus();
            if (status == null)
            {
                Debug.LogError("[MrModeSwitcher] Status result null");
                return;
            }
            if (status.Data.BluetoothStatus == ConnectionStatus.Connected
            && status.Data.WifiStatus == ConnectionStatus.Connected)
            {
                Debug.Log("[MrModeSwitcher] BT WiFi OK");
            }
            else if (!status.Data.SpacesAvailable)
            {
                Debug.LogError("[MrModeSwitcher] Spaces not Available");
                return;
            }
            else
            {
                Debug.LogError("[MrModeSwitcher] not Connected");
                return;
            }

            var mrMode = _mirzaPlugin.GetSpacesModeStatus();
            if (mrMode == null)
            {
                Debug.LogError("[MrModeSwitcher] MRMode result null");
                return;
            }
            if (mrMode.Data == SpacesModeStatus.On)
            {
                Debug.Log("[MrModeSwitcher] MR Mode ON");
                _isMRMode = true;
                return;
            }
            else
            {
                Debug.Log("[MrModeSwitcher] MR Mode OFF");
            }
            var modeOn = _mirzaPlugin.SpacesModeOn();
            if (modeOn == null)
            {
                Debug.LogError("[MrModeSwitcher] Mode On result null");
                return;
            }
            Debug.Log("[MrModeSwitcher] Change MR Mode OK");
            _isMRMode = true;
        }

        private void OnOpenXRAvailable()
        {
            Debug.Log("[MrModeSwitcher] OnOpenXRAvailable");
            if (_isMiRZAServiceConnected)
            {
                if (_isMRMode && changeScene)
                {
                    ChangeScene();
                }

                if (!_isMRMode)
                {
                    try
                    {
                        MRModeSequence();
                    }
                    catch (Exception ex)
                    {
                        Debug.LogError(ex.ToString());
                    }
                    return;
                }
            }
            else
            {
                Debug.LogError("[MrModeSwitcher] MiRZA Service not connected !");
            }
            _dynamicOpenXRLoader.StartOpenXR();
        }

        private void OnOpenXRNotAvailable()
        {
            Debug.Log("[MrModeSwitcher] OnOpenXRNotAvailable");
            _dynamicOpenXRLoader.StopOpenXR();
        }

        private IEnumerator CheckGlassStatus()
        {
            while (true)
            {
                if (_spacesGlassStatus.GlassConnectionState != _glassConnectionState)
                {
                    _glassConnectionState = _spacesGlassStatus.GlassConnectionState;
                    OnConnectionStateChanged.Invoke(_glassConnectionState);
                }
                yield return new WaitForSeconds(1);
            }
        }

        public void ChangeScene()
        {
            SceneManager.LoadScene(sceneName);
        }
    }
}
#endif