﻿#if USING_MIRZA_SDK

using System;
using com.nttqonoq.devices.android.mirzalibrary;
using UnityEngine;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Runtime
{
    public class MirzaPluginSimulator : IMirzaPlugin
    {
        private bool _isMonitoring = false;
        private Result<string> _versionConsistencyResult;
        private Result<GlassStatus> _glassStatusResult;
        private Result<int> _batteryLevelResult;
        private Result<ChargeStatus> _chargeStatusResult;
        private Result<int> _spacesModeOnResult;
        private Result<int> _spacesModeOffResult;
        private Result<SpacesModeStatus> _spacesModeStatusResult;

        public Result<string> VersionConsistencyResult { set => _versionConsistencyResult = value; }
        public Result<GlassStatus> GlassStatusResult { set => _glassStatusResult = value; }
        public Result<int> BatteryLevelResult { set => _batteryLevelResult = value; }
        public Result<ChargeStatus> ChargeStatusResult { set => _chargeStatusResult = value; }
        public Result<int> SpacesModeOnResult { set => _spacesModeOnResult = value; }
        public Result<int> SpacesModeOffResult { set => _spacesModeOffResult = value; }
        public Result<SpacesModeStatus> SpacesModeStatusResult { set => _spacesModeStatusResult = value; }

        public event Action OnSpacesModeOn;
        public event Action OnSpacesModeOff;

        public void StartMonitoring()
        {
            Debug.Log("(Simulator) StartMonitoring() is called.");
            _isMonitoring = true;
        }

        public void StopMonitoring()
        {
            Debug.Log("(Simulator) StopMonitoring() is called.");
            _isMonitoring = false;
        }

        public void SetLogEnable(bool enabled)
        {
            Debug.Log("(Simulator) SetLogEnable() is called. enabled: " + enabled);
        }

        public Result<string> CheckVersionConsistency()
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) CheckVersionConsistency() is called.");
                return _versionConsistencyResult;
            }
            else
            {
                Debug.LogError("(Simulator) CheckVersionConsistency() is called, but monitoring is not started. Call StartMonitoring() first.");
                return new Result<string>(State.ServiceNotConnected, null, null);
            }
        }

        public Result<GlassStatus> GetGlassStatus()
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetGlassStatus() is called.");
                return _glassStatusResult;
            }
            else
            {
                Debug.LogError("(Simulator) GetGlassStatus() is called, but monitoring is not started. Call StartMonitoring() first.");
                return null;
            }
        }

        public void GetGlassStatusAsync(Action<Result<GlassStatus>> onResult)
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetGlassStatusAsync() is called.");
                onResult?.Invoke(_glassStatusResult);
            }
            else
            {
                Debug.LogError("(Simulator) GetGlassStatusAsync() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }

        public Result<int> GetBatteryLevel()
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetBatteryLevel() is called.");
                return _batteryLevelResult;
            }
            else
            {
                Debug.LogError("(Simulator) GetBatteryLevel() is called, but monitoring is not started. Call StartMonitoring() first.");
                return new Result<int>(State.ServiceNotConnected, 0, null);
            }
        }

        public void GetBatteryLevelAsync(Action<Result<int>> onResult)
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetBatteryLevelAsync() is called.");
                onResult?.Invoke(_batteryLevelResult);
            }
            else
            {
                Debug.LogError("(Simulator) GetBatteryLevelAsync() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }

        public Result<ChargeStatus> GetChargeStatus()
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetChargeStatus() is called.");
                return _chargeStatusResult;
            }
            else
            {
                Debug.LogError("(Simulator) GetChargeStatus() is called, but monitoring is not started. Call StartMonitoring() first.");
                return new Result<ChargeStatus>(State.ServiceNotConnected, ChargeStatus.Off, null);
            }
        }

        public void GetChargeStatusAsync(Action<Result<ChargeStatus>> onResult)
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetChargeStatusAsync() is called.");
                onResult?.Invoke(_chargeStatusResult);
            }
            else
            {
                Debug.LogError("(Simulator) GetChargeStatusAsync() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }

        public Result<int> SpacesModeOn()
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) SpacesModeOn() is called.");
                OnSpacesModeOn?.Invoke();
                return _spacesModeOnResult;
            }
            else
            {
                Debug.LogError("(Simulator) SpacesModeOn() is called, but monitoring is not started. Call StartMonitoring() first.");
                return new Result<int>(State.ServiceNotConnected, 0, null);
            }
        }

        public void SpacesModeOnAsync(Action<Result<int>> onResult)
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) SpacesModeOnAsync() is called.");
                OnSpacesModeOn?.Invoke();
                onResult?.Invoke(_spacesModeOnResult);
            }
            else
            {
                Debug.LogError("(Simulator) SpacesModeOnAsync() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }

        public Result<int> SpacesModeOff()
        {
            if (_isMonitoring)
            {

                Debug.Log("(Simulator) SpacesModeOff() is called.");
                OnSpacesModeOff?.Invoke();
                return _spacesModeOffResult;
            }
            else
            {
                Debug.LogError("(Simulator) SpacesModeOff() is called, but monitoring is not started. Call StartMonitoring() first.");
                return new Result<int>(State.ServiceNotConnected, 0, null);
            }
        }

        public void SpacesModeOffAsync(Action<Result<int>> onResult)
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) SpacesModeOffAsync() is called.");
                OnSpacesModeOff?.Invoke();
                onResult?.Invoke(_spacesModeOffResult);
            }
            else
            {
                Debug.LogError("(Simulator) SpacesModeOffAsync() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }

        public Result<SwitchMicrophoneStatus> SwitchMicrophone(MicMode wearingMicMode, MicMode frontMicMode, MixMode mixMode)
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) SwitchMicrophone() is called.");

                var switchMicrophoneStatus = MirzaPluginUtility.SwitchMicrophoneStatus(wearingMicMode, frontMicMode, mixMode);
                var result = new Result<SwitchMicrophoneStatus>(State.Success, switchMicrophoneStatus, MirzaPluginUtility.ErrorMessage);
                return result;
            }
            else
            {
                Debug.LogError("(Simulator) SwitchMicrophone() is called, but monitoring is not started. Call StartMonitoring() first.");
                return null;
            }
        }

        public void SwitchMicrophoneAsync(MicMode wearingMicMode, MicMode frontMicMode, MixMode mixMode, Action<Result<SwitchMicrophoneStatus>> onResult)
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) SwitchMicrophoneAsync() is called.");
                var switchMicrophoneStatus = MirzaPluginUtility.SwitchMicrophoneStatus(wearingMicMode, frontMicMode, mixMode);
                var result = new Result<SwitchMicrophoneStatus>(State.Success, switchMicrophoneStatus, MirzaPluginUtility.ErrorMessage);

                onResult?.Invoke(result);
            }
            else
            {
                Debug.LogError("(Simulator) SwitchMicrophoneAsync() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }

        public Result<SpacesModeStatus> TransitionToMirzaAppInSpacesMode()
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) TransitionToMirzaAppInSpacesMode() is called.");
                return _spacesModeStatusResult;
            }
            else
            {
                Debug.LogError("(Simulator) TransitionToMirzaAppInSpacesMode() is called, but monitoring is not started. Call StartMonitoring() first.");
                return new Result<SpacesModeStatus>(State.ServiceNotConnected, SpacesModeStatus.Off, null);
            }
        }

        public string GetVersion()
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetVersion() is called.");
                return MirzaPluginUtility.Version;
            }
            else
            {
                Debug.LogError("(Simulator) GetVersion() is called, but monitoring is not started. Call StartMonitoring() first.This returns default value, 1.0.0.");
                return "1.0.0";
            }
        }

        public Result<SpacesModeStatus> GetSpacesModeStatus()
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetSpacesModeStatus() is called.");
                return _spacesModeStatusResult;
            }
            else
            {
                Debug.LogError("(Simulator) GetSpacesModeStatus() is called, but monitoring is not started. Call StartMonitoring() first.");
                return new Result<SpacesModeStatus>(State.ServiceNotConnected, SpacesModeStatus.Off, null);
            }
        }

        public void GetSpacesModeStatusAsync(Action<Result<SpacesModeStatus>> onResult)
        {
            if (_isMonitoring)
            {
                Debug.Log("(Simulator) GetSpacesModeStatusAsync() is called.");
                onResult.Invoke(_spacesModeStatusResult);
            }
            else
            {
                Debug.LogError("(Simulator) GetSpacesModeStatusAsync() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }

        public event Action<ServiceState> OnServiceStateChanged;
        public event Action<GlassStatus> OnGlassStatusChanged;
        public event Action<int> OnBatteryLevelChanged;
        public event Action<SpacesModeStatus> OnSpacesModeStatusChanged;
        public event Action<DisplayStatus> OnDisplayStatusChanged;
        public event Action OnPowerOffChanged;
        public event Action<GlassTouchGestureStatus> OnGlassTouchGestureStatusChanged;

        public void SimulateServiceStateChanged(ServiceState serviceState)
        {
            if (_isMonitoring)
            {
                OnServiceStateChanged?.Invoke(serviceState);
            }
            else
            {
                Debug.LogError("(Simulator) SimulateServiceStateChanged() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }
        public void SimulateGlassStatusChanged(GlassStatus glassStatus)
        {
            if (_isMonitoring)
            {
                OnGlassStatusChanged?.Invoke(glassStatus);
            }
            else
            {
                Debug.LogError("(Simulator) SimulateGlassStatusChanged() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }
        public void SimulateBatteryLevelChanged(int batteryLevel)
        {
            if (_isMonitoring)
            {
                OnBatteryLevelChanged?.Invoke(batteryLevel);
            }
            else
            {
                Debug.LogError("(Simulator) SimulateBatteryLevelChanged() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }
        public void SimulateSpacesModeStatusChanged(SpacesModeStatus spacesModeStatus)
        {
            if (_isMonitoring)
            {
                OnSpacesModeStatusChanged?.Invoke(spacesModeStatus);
            }
            else
            {
                Debug.LogError("(Simulator) SimulateSpacesModeStatusChanged() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }
        public void SimulateDisplayStatusChanged(DisplayStatus displayStatus)
        {
            if (_isMonitoring)
            {
                OnDisplayStatusChanged?.Invoke(displayStatus);
            }
            else
            {
                Debug.LogError("(Simulator) SimulateDisplayStatusChanged() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }
        public void SimulatePowerOffChanged()
        {
            if (_isMonitoring)
            {
                OnPowerOffChanged?.Invoke();
            }
            else
            {
                Debug.LogError("(Simulator) SimulatePowerOffChanged() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }
        public void SimulateGlassTouchGestureStatusChanged(GlassTouchGestureStatus glassTouchGestureStatus)
        {
            if (_isMonitoring)
            {
                OnGlassTouchGestureStatusChanged?.Invoke(glassTouchGestureStatus);
            }
            else
            {
                Debug.LogError("(Simulator) SimulateGlassTouchGestureStatusChanged() is called, but monitoring is not started. Call StartMonitoring() first.");
            }
        }
    }
}
#endif