﻿#if USING_MIRZA_SDK && UNITY_EDITOR
using com.nttqonoq.devices.android.mirzalibrary;
using System;
using UnityEngine;
using TouchType = com.nttqonoq.devices.android.mirzalibrary.TouchType;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Runtime
{
    public class MirzaDeviceEventSimulator : MonoBehaviour
    {

        public readonly string[] _serviceStateOptions = ServiceState.GetNames(typeof(ServiceState));
        public readonly string[] _connectionStatusOptions = ConnectionStatus.GetNames(typeof(ConnectionStatus));
        public readonly string[] _spacesModeStatusOptions = SpacesModeStatus.GetNames(typeof(SpacesModeStatus));
        public readonly string[] _displayStatusOptions = DisplayStatus.GetNames(typeof(DisplayStatus));
        public readonly string[] _touchTypeOptions = TouchType.GetNames(typeof(TouchType));
        public readonly string[] _touchOperationOptions = TouchOperation.GetNames(typeof(TouchOperation));
        public readonly string[] _stateOptions = State.GetNames(typeof(State));
        public readonly string[] _chargeStatusOptions = ChargeStatus.GetNames(typeof(ChargeStatus));

        private MirzaPluginSimulator _mirzaPluginSimulator;

        [SerializeField]
        private int selectedStateIndex;

        [SerializeField]
        private int selectedBluetoothStatusIndex;

        [SerializeField]
        private int selectedWifiStatusIndex;

        [SerializeField]
        private bool spacesAvailable;

        [SerializeField]
        private int batteryLevel;

        [SerializeField]
        private int selectedSpacesModeStatusIndex;

        [SerializeField]
        private int selectedDisplayStatusIndex;

        [SerializeField]
        private int selectedTouchTypeIndex;

        [SerializeField]
        private int selectedTouchOperationIndex;

        [SerializeField]
        private int xCoordinate;

        [SerializeField]
        private int yCoordinate;

        [SerializeField]
        private int movement;

        [SerializeField]
        private int stateIndex;

        [SerializeField]
        private string versionConsistencyResultData;

        [SerializeField]
        private int chargeStatusIndex;


        // Getで渡すResultで共通のため保持
        private State currentState;
        // Event/Get 両対応のため保持
        private GlassStatus currentGlassStatus;
        private SpacesModeStatus currentSpacesModeStatus;

        /// <summary>
        /// https://www.devices.nttqonoq.com/developer/doc/mirza-library/mirzalibrary/#spacesmodeon
        /// 有効化結果オブジェクト。関連データはなく問題があった場合はエラーメッセージが返却されます。
        /// 固定値
        /// </summary>
        private readonly int spacesModeOnStatus = 0;
        /// <summary>
        /// https://www.devices.nttqonoq.com/developer/doc/mirza-library/mirzalibrary/#spacesmodeoff
        /// 無効化結果オブジェクト。関連データはなく問題があった場合はエラーメッセージが返却されます。
        /// 固定値
        /// </summary>
        private readonly int spacesModeOffStatus = 0;

        public State CurrentState { set => currentState = value; }

        public void UpdateVersionConsistencyResult()
        {
            var result = new Result<string>(currentState, versionConsistencyResultData, MirzaPluginUtility.ErrorMessage);
            _mirzaPluginSimulator.VersionConsistencyResult = result;
        }

        public void UpdateGlassStatusResult()
        {
            ConnectionStatus bluetoothStatus = (ConnectionStatus)Enum.Parse(typeof(ConnectionStatus), _connectionStatusOptions[selectedBluetoothStatusIndex]);
            ConnectionStatus wifiStatus = (ConnectionStatus)Enum.Parse(typeof(ConnectionStatus), _connectionStatusOptions[selectedWifiStatusIndex]);
            bool spacesAvailable = this.spacesAvailable;

            currentGlassStatus = MirzaPluginUtility.CreateGlassStatusInstance(bluetoothStatus, wifiStatus, spacesAvailable);

            var result = new Result<GlassStatus>(currentState, currentGlassStatus, MirzaPluginUtility.ErrorMessage);
            _mirzaPluginSimulator.GlassStatusResult = result;
        }

        public void UpdateBatteryLevelResult()
        {
            var result = new Result<int>(currentState, batteryLevel, MirzaPluginUtility.ErrorMessage);
            _mirzaPluginSimulator.BatteryLevelResult = result;
        }

        public void UpdateChargeStatusResult()
        {
            ChargeStatus chargeStatus = (ChargeStatus)Enum.Parse(typeof(ChargeStatus), _chargeStatusOptions[chargeStatusIndex]);
            var result = new Result<ChargeStatus>(currentState, chargeStatus, MirzaPluginUtility.ErrorMessage);
            _mirzaPluginSimulator.ChargeStatusResult = result;
        }

        public void UpdateSpacesModeStatusResult()
        {
            SpacesModeStatus spacesModeStatus = (SpacesModeStatus)Enum.Parse(typeof(SpacesModeStatus), _spacesModeStatusOptions[selectedSpacesModeStatusIndex]);
            currentSpacesModeStatus = spacesModeStatus;
            var result = new Result<SpacesModeStatus>(currentState, currentSpacesModeStatus, MirzaPluginUtility.ErrorMessage);
            _mirzaPluginSimulator.SpacesModeStatusResult = result;
        }

        public void UpdateSpacesModeOnResult()
        {
            var result = new Result<int>(currentState, spacesModeOnStatus, MirzaPluginUtility.ErrorMessage);
            _mirzaPluginSimulator.SpacesModeOnResult = result;
        }

        public void UpdateSpacesModeOffResult()
        {
            var result = new Result<int>(currentState, spacesModeOffStatus, MirzaPluginUtility.ErrorMessage);
            _mirzaPluginSimulator.SpacesModeOffResult = result;
        }


        private void Awake()
        {
            IMirzaPlugin _mirzaPlugin = MirzaPluginProvider.MirzaPlugin;
            _mirzaPluginSimulator = _mirzaPlugin as MirzaPluginSimulator;

            _mirzaPluginSimulator.OnSpacesModeOn  += () =>
            {
                if(currentSpacesModeStatus == SpacesModeStatus.Off)
                {
                    currentSpacesModeStatus = SpacesModeStatus.On;
                    selectedSpacesModeStatusIndex = (int)SpacesModeStatus.On;
                    UpdateSpacesModeStatusResult();
                    HandleSpacesModeStatusChanged();
                }
            };
            _mirzaPluginSimulator.OnSpacesModeOff += () =>
            {
                if(currentSpacesModeStatus == SpacesModeStatus.On)
                {
                    currentSpacesModeStatus = SpacesModeStatus.Off;
                    selectedSpacesModeStatusIndex = (int)SpacesModeStatus.Off;
                    UpdateSpacesModeStatusResult();
                    HandleSpacesModeStatusChanged();
                }
            };

            UpdateVersionConsistencyResult();
            UpdateGlassStatusResult();
            UpdateBatteryLevelResult();
            UpdateChargeStatusResult();
            UpdateSpacesModeOnResult();
            UpdateSpacesModeOffResult();
            UpdateSpacesModeStatusResult();
        }


        public void HandleServiceStateChanged(ServiceState state)
        {
            _mirzaPluginSimulator?.SimulateServiceStateChanged(state);
        }

        public void HandleGlassStatusChanged()
        {
            _mirzaPluginSimulator?.SimulateGlassStatusChanged(currentGlassStatus);
        }

        public void HandleBatteryLevelChanged(int batteryLevel)
        {
            _mirzaPluginSimulator?.SimulateBatteryLevelChanged(batteryLevel);
        }


        public void HandleSpacesModeStatusChanged()
        {
            _mirzaPluginSimulator?.SimulateSpacesModeStatusChanged(currentSpacesModeStatus);
        }

        public void HandleDisplayStatusChanged(DisplayStatus displayStatus)
        {
            _mirzaPluginSimulator?.SimulateDisplayStatusChanged(displayStatus);
        }

        public void HandlePowerOffChanged()
        {
            _mirzaPluginSimulator?.SimulatePowerOffChanged();
        }

        public void HandleGlassTouchGestureStatusChanged(TouchType touchType, TouchOperation touchOperation, int xCoordinate, int yCoordinate, int movement)
        {

            GlassTouchGestureStatus status = MirzaPluginUtility.CreateGlassTouchGestureStatusInstance(touchType, touchOperation, xCoordinate, yCoordinate, movement);
            _mirzaPluginSimulator?.SimulateGlassTouchGestureStatusChanged(status);
        }

    }
}
#endif