using UnityEngine;
using Unity.XR.CoreUtils;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Runtime
{
    /// <summary>
    /// FloatingPanel is a MonoBehaviour that allows a UI panel to follow the camera's position and orientation in a 3D space.
    /// </summary>
    public class FloatingPanel : MonoBehaviour
    {
        [Tooltip("If true, the panel will follow the camera's position and orientation. If false, it will remain stationary.")]
        [SerializeField]
        private bool _followCamera = true;

        [Tooltip("Distance from the camera where the panel will be positioned. This is the distance along the camera's forward direction.")]
        [SerializeField]
        public float _targetDistance = 2.0f;

        [Tooltip("Smoothness of the panel's movement towards the target position. A value of 0.0 means no smoothing, while 1.0 means instant movement.")]
        [SerializeField]
        [Range(0.0f, 1.0f)]
        private float _movementSmoothness = 0.2f;

        [Tooltip("Vertical bias for the camera's field of view. Adjusts how much vertical angle is considered when positioning the panel.")]
        [SerializeField]
        private float _verticalBias = 0.8f;

        [Tooltip("Horizontal bias for the camera's field of view. Adjusts how much horizontal angle is considered when positioning the panel.")]
        [SerializeField]
        private float _horizontalBias = 0.5f;

        private Transform _cameraTransform;
        private Camera _camera;

        /// <summary>
        /// Gets or sets a value indicating whether the panel should follow the camera.
        /// </summary>
        public bool FollowCamera
        {
            get => _followCamera;
            set => _followCamera = value;
        }

        private void Start()
        {
            var xrOrigin = FindFirstObjectByType<XROrigin>(FindObjectsInactive.Include);
            if (xrOrigin == null)
            {
                Debug.LogWarning("XROrigin not found. FloatingPanel will not follow the camera.");
                return;
            }
            _camera = xrOrigin.Camera;
            _cameraTransform = _camera.transform;
        }

        private void Update()
        {
            if (!_followCamera || _cameraTransform == null || _camera == null)
            {
                return;
            }

            var cameraPosition = _cameraTransform.position;
            var cameraForward = _cameraTransform.forward;
            var panelPosition = transform.position;
            var fov = _camera.fieldOfView;

            var direction = (panelPosition - cameraPosition).normalized;
            var targetPosition = cameraPosition + cameraForward * _targetDistance;
            var targetDirection = (targetPosition - cameraPosition).normalized;
            var eulerAngles = Quaternion.LookRotation(direction).eulerAngles;
            var targetEulerAngles = Quaternion.LookRotation(targetDirection).eulerAngles;

            // Adjust the euler angles based on the camera's field of view and biases.
            eulerAngles.x += GetClampedRotationDelta(targetEulerAngles.x - eulerAngles.x, fov * _verticalBias);
            eulerAngles.y += GetClampedRotationDelta(targetEulerAngles.y - eulerAngles.y, fov * _horizontalBias * _camera.aspect);

            targetPosition = cameraPosition + (Quaternion.Euler(eulerAngles) * Vector3.forward * _targetDistance);

            // Smoothly move the panel towards the target position and rotation.
            var nextPosition = Vector3.Lerp(panelPosition, targetPosition, _movementSmoothness);
            var nextRotation = Quaternion.Lerp(transform.rotation, Quaternion.LookRotation(nextPosition - cameraPosition), _movementSmoothness);
            transform.SetPositionAndRotation(nextPosition, nextRotation);
        }

        /// <summary>
        /// Clamps the rotation delta to a specified threshold.
        /// </summary>
        private float GetClampedRotationDelta(float angle, float threshold)
        {
            angle = (540f + angle) % 360f - 180f;

            var absAngle = Mathf.Abs(angle);
            if (absAngle > threshold)
            {
                return -angle / absAngle * (threshold - absAngle);
            }

            return 0f;
        }
    }
}