#if USING_MIRZA_SDK
using com.nttqonoq.devices.android.mirzalibrary;
#endif
using System.Collections;
using TMPro;
using Unity.Profiling;
using UnityEngine;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Runtime
{
    public class DebugHUD : MonoBehaviour
    {
        public enum DisplayMode
        {
            None,
            Left
        }

        [SerializeField] private TMP_Text _fpsText;

        [SerializeField] private TMP_Text _cpuText;

        [SerializeField] private TMP_Text _gpuText;

        [SerializeField] private TMP_Text _usedMomoryText;

        [SerializeField] private TMP_Text _batteryText;

        [SerializeField] private TMP_Text _touchTypeText;

        [SerializeField] private TMP_Text _touchOperationText;

        [SerializeField] private TMP_Text _touchCoordinateText;

        private DisplayMode _displayMode = DisplayMode.None;

        private Canvas _canvas;

        private readonly FrameTiming[] _frameTimings = new FrameTiming[1];

        private ProfilerRecorder _systemMemoryRecorder;

        private int _bufferSize = 32;

        private float[] _cpuBuffer;
        private float[] _gpuBuffer;
        private int _bufferIndex;

        private int _lastBatteryLevel = -1;

#if USING_MIRZA_SDK
        private IMirzaPlugin _mirzaPlugin;
#endif

        private bool _enableMirzaMonitoring;

        public bool EnableMirzaMonitoring
        {
            get => _enableMirzaMonitoring;
            set => _enableMirzaMonitoring = value;
        }

        public float UpdateInterval { get; set; } = 1f;

        public int FpsBufferSize
        {
            get => _bufferSize;
            set
            {
                if (value > 0)
                {
                    _bufferSize = value;
                    _cpuBuffer = null;
                    _gpuBuffer = null;
                    _bufferIndex = 0;
                }
            }
        }

        private void Awake()
        {
            _canvas = GetComponentInChildren<Canvas>();
            _canvas.gameObject.SetActive(false);
        }

        private void Start()
        {
#if !UNITY_EDITOR && USING_MIRZA_SDK
            if (_enableMirzaMonitoring)
            {
                _mirzaPlugin = new MirzaPlugin();
                _mirzaPlugin.OnGlassTouchGestureStatusChanged += OnGlassTouchGestureStatusChanged;
                _mirzaPlugin.SetLogEnable(true);
                _mirzaPlugin.StartMonitoring();
            }
#endif
        }

        private void OnDestroy()
        {
#if USING_MIRZA_SDK
            if (_mirzaPlugin != null)
            {
                _mirzaPlugin.StopMonitoring();
                _mirzaPlugin.OnGlassTouchGestureStatusChanged -= OnGlassTouchGestureStatusChanged;
            }
#endif
        }

#if USING_MIRZA_SDK
        private void OnGlassTouchGestureStatusChanged(GlassTouchGestureStatus status)
        {

            if (_mirzaPlugin == null)
            {
                return;
            }

            if (status == null)
            {
                return;
            }

            //Debug.Log($"Type: {status.Type}, Operation: {status.Operation}, X: {status.XCoordinate}, Y: {status.YCoordinate}, Movement, {status.Movement}");

            _touchTypeText.text = status.Type.ToString();
            _touchOperationText.text = status.Operation.ToString();
            _touchCoordinateText.text = $"{status.XCoordinate},{status.YCoordinate},{status.Movement}";
        }
#endif

        private void OnEnable()
        {
            _systemMemoryRecorder = ProfilerRecorder.StartNew(ProfilerCategory.Memory, "System Used Memory");
            StartCoroutine(UpdateDisplayCoroutine());
#if USING_MIRZA_SDK
            StartCoroutine(UpdateBatteryLevelCoroutine());
#endif
        }

        private void OnDisable()
        {
            _systemMemoryRecorder.Dispose();
            StopCoroutine(UpdateDisplayCoroutine());
#if USING_MIRZA_SDK
            StopCoroutine(UpdateBatteryLevelCoroutine());
#endif
        }

        private IEnumerator UpdateDisplayCoroutine()
        {
            while (true)
            {
                if (_canvas == null || !_canvas.gameObject.activeSelf)
                {
                    yield return null;
                }

                if (_cpuBuffer != null)
                {
                    var averageCpuFrameTime = 0f;
                    for (var i = 0; i < _cpuBuffer.Length; i++)
                    {
                        averageCpuFrameTime += _cpuBuffer[i];
                    }

                    averageCpuFrameTime /= _bufferSize;

                    if (averageCpuFrameTime > 0)
                    {
                        var fps = 1f / (averageCpuFrameTime / 1000f);
                        _fpsText.text = $"{fps:0.0}";
                    }

                    _cpuText.text = $"{averageCpuFrameTime:0.0}ms";
                }

                if (_gpuBuffer != null)
                {
                    var averageGpuFrameTime = 0f;
                    for (var i = 0; i < _gpuBuffer.Length; i++)
                    {
                        averageGpuFrameTime += _gpuBuffer[i];
                    }

                    averageGpuFrameTime /= _bufferSize;

                    _gpuText.text = $"{averageGpuFrameTime:0.0}ms";
                }

                // Memory
                var usedMemory = _systemMemoryRecorder.LastValue / (1024f * 1024f);
                _usedMomoryText.text = $"{usedMemory:0.0}MB";

                yield return new WaitForSeconds(UpdateInterval);
            }
        }

#if USING_MIRZA_SDK
        private IEnumerator UpdateBatteryLevelCoroutine()
        {
            while (true)
            {
                if (_mirzaPlugin == null)
                {
                    _lastBatteryLevel = -1;
                    yield return new WaitForSeconds(10f);
                }

                var level =  _mirzaPlugin?.GetBatteryLevel();
                if (level != null)
                {
                    _lastBatteryLevel = level.Data;
                    _batteryText.text = $"{_lastBatteryLevel}%";
                }
                yield return new WaitForSeconds(10f);
            }
        }
#endif

        private void Update()
        {
            FrameTimingManager.CaptureFrameTimings();
            var numFrames = FrameTimingManager.GetLatestTimings((uint)_frameTimings.Length, _frameTimings);
            if (numFrames != 0)
            {
                if(_cpuBuffer == null)
                {
                    _cpuBuffer = new float[_bufferSize];
                    _gpuBuffer = new float[_bufferSize];
                    _bufferIndex = 0;
                }

                var cpuFrameTime = (float)(_frameTimings[0].cpuFrameTime);
                _cpuBuffer[_bufferIndex] = cpuFrameTime;

                var gpuFrameTime = (float)(_frameTimings[0].gpuFrameTime);
                _gpuBuffer[_bufferIndex] = gpuFrameTime;

                _bufferIndex = (_bufferIndex + 1) % _bufferSize;
            }
        }

        public DisplayMode CurrentDisplayMode => _displayMode;

        public DisplayMode ToggleDisplayMode()
        {
            if (_displayMode == DisplayMode.None)
            {
                _displayMode = DisplayMode.Left;
                _canvas.gameObject.SetActive(true);
                ((RectTransform)_canvas.transform).anchoredPosition3D = new Vector3(-0.04f, 0.0f, 0.25f);
            }
            else
            {
                _displayMode = DisplayMode.None;
                _canvas.gameObject.SetActive(false);
            }

            return _displayMode;
        }
    }
}
