﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Unity.XR.CoreUtils;
using UnityEditor;
using UnityEditor.PackageManager;
using UnityEditor.PackageManager.Requests;
using UnityEditor.PackageManager.UI;
using UnityEngine;
using UnityEngine.Rendering;
using Object = UnityEngine.Object;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public class Validator
    {
        public static readonly string MirzaLibraryPackageName = "com.nttqonoq.devices.android.mirzalibrary";
        public static readonly string InteractionsPackageName = "com.qualcomm.qcht.unity.interactions";
        public static readonly string SpacesPackageName = "com.qualcomm.snapdragon.spaces";
        public static readonly string ConfigToolPackageName = "com.nttqonoq.devices.android.mirza.configuration-tool";
        public static readonly string OpenXRPluginPackageName = "com.unity.xr.openxr";
        public static readonly string NewtonsoftJsonPackageName = "com.unity.nuget.newtonsoft-json";

        public bool IsDone { get; private set; }

        private ValidationReport _report;

        private ListRequest _listRequest;

        private Action<ValidationReport> _onComplete;

        private bool _isDualRenderFusionMode;

        public Validator(bool isDualRenderFusionMode = true)
        {
            _isDualRenderFusionMode = isDualRenderFusionMode;
        }

        public void Validate(Action<ValidationReport> onComplete)
        {
            _onComplete = onComplete;

            IsDone = false;

            _report = new ValidationReport();
            _report.CreateDate = DateTimeOffset.Now.ToString();

            ValidateProjectParams();

            ValidateSceneParams();

            ValidateSimulatorParams();

            ValidateTemplatesParams();

            ValidateSamplesParams();

            ValidateDebugParams();
        }

        private void ValidateProjectParams()
        {
            var p = _report.ProjectParams;

            p.Packages = new List<PackageParam>
            {
                new() { Name = MirzaLibraryPackageName, DisplayName = "MiRZA Library", MinRequiredVersion = "1.0.0" },
                new() { Name = InteractionsPackageName, DisplayName = "QCHT Unity Interactions", MinRequiredVersion = "1.0.0" },
                new() { Name = SpacesPackageName, DisplayName = "Snapdragon Spaces", MinRequiredVersion = "1.0.0" },
                new() { Name = ConfigToolPackageName, DisplayName = "MiRZA Configuration Tool", },
                new() { Name = OpenXRPluginPackageName, DisplayName = "OpenXR Plugin", MinRequiredVersion = "1.0.0", ExpectedVersion = "1.10.0" },
                new() { Name = NewtonsoftJsonPackageName, DisplayName = "Newtonsoft Json" },
            };

            p.Assets = new List<AssetParam>
            {
            };

            // Check Android Version

            p.AndroidMinimumApiLevel = new Param()
            {
                Exists = true,
                Actual = PlayerSettings.Android.minSdkVersion.ToString(),
            };

            p.AndroidTargetApiLevel = new Param()
            {
                Exists = true,
                Actual = PlayerSettings.Android.targetSdkVersion.ToString(),
            };

            // Check Unity Version
            p.UnityVersion = new Param
            {
                Exists = true,
                Actual = Application.unityVersion,
            };
            var minimumVersion = new Version(2022, 3, 16, 1);
            var overVersion = new Version(6000, 1, 0, 0);
            var currentVersion = GetUnityVersion();
            if (currentVersion != null)
            {
                if (currentVersion < minimumVersion)
                {
                    p.UnityVersion.Expected = "2022.3.16f1";
                    p.UnityVersion.Message = "Unity version is not supported. Please use 2022.3 LTS or Unity 6.0 LTS.";
                }
                else if (currentVersion >= overVersion)
                {
                    p.UnityVersion.Expected = "6000.0.0f1";
                    p.UnityVersion.Message = "Unity version is not supported. Please use 2022.3 LTS or Unity 6.0 LTS.";
                }
            }

            // Check Active Input Handling
            p.ActiveInputHandling = ValidationUtility.GetActiveInputHandling(_isDualRenderFusionMode);

            // Check Allow Unsafe Code
            p.AllowUnsafeCode = ValidationUtility.GetAllowUnsafeCode();

            // Check Platform
            p.Platform = new Param()
            {
                Exists = true,
                Expected = "Android",
                Actual = EditorUserBuildSettings.activeBuildTarget.ToString(),
            };

            p.InitializeXROnStartup = ValidationUtility.GetInitializeXrOnStartup(BuildTargetGroup.Android, _isDualRenderFusionMode);

            p.OpenXRPluginEnabled = new Param()
            {
                Exists = true,
                Actual = Utility.IsOpenXrPluginEnabled(BuildTargetGroup.Android).ToString(),
                Expected = true.ToString()
            };

            p.SnapdragonSpacesFeatureGroupEnabled = ValidationUtility.GetSpacesFeatureEnabled();

            p.BaseRuntimeFeatureEnabled = ValidationUtility.GetBaseRuntimeFeature(BuildTargetGroup.Android);

            p.LaunchAppOnViewer = ValidationUtility.GetLaunchAppOnViewer(_isDualRenderFusionMode);
            p.LaunchControllerOnHost = ValidationUtility.GetLaunchControllerOnHost(_isDualRenderFusionMode);
            p.ExportHeadless = ValidationUtility.GetExportHeadless(_isDualRenderFusionMode);

            p.DualRenderFusionFeatureEnabled = ValidationUtility.GetDualRenderFusionFeature();

            p.SnapdragonSpacesExperimentalFeatureGroupEnabled = ValidationUtility.GetSpacesExperimentalFeatureEnabled();

            p.IgnoreOpenXRVersion = ValidationUtility.GetIgnoreOpenXRVersion();

            ValidateSamples();

            // Get package list
            _listRequest = Client.List();
            EditorApplication.update += CheckListRequest;
        }

        private void ValidateSceneParams()
        {
            var p = _report.SceneParams;

            p.ARSession = ValidationUtility.GetARSession();

            p.XROrigin = ValidationUtility.GetXROrigin();

            p.ARCameraManager = ValidationUtility.GetARCameraManager();

            p.ARCameraBackground = ValidationUtility.GetARCameraBackground();

            p.DynamicOpenXRLoader = ValidationUtility.GetDynamicOpenXRLoader();

            p.SpacesHostView = ValidationUtility.GetSpacesHostView();

            p.DualRenderFusionController = ValidationUtility.GetDualRenderFusionController();

            p.RenderCamera = ValidationUtility.GetRenderCamera();

            p.SpacesLifecycleEvents = ValidationUtility.GetSpacesLifecycleEvents();

            p.XRInteractionManager = ValidationUtility.GetXRInteractionManager();

            p.ConfiguredInputActionManager = ValidationUtility.GetConfiguredInputActionManager();

            p.QCHTXROrigin = ValidationUtility.GetQCHTXROrigin();

            p.SpacesXRSimulator = ValidationUtility.GetSpacesXRSimulator();

            p.HandTrackingFeatureEnabled = ValidationUtility.GetHandTrackingFeature();

            p.SpacesPermissionHelper = ValidationUtility.GetSpacesPermissionHelper();

            p.MRModeAutoSwitcher = ValidationUtility.GetMRModeAutoSwitcher();

            ValidateCameras();
        }

        private void ValidateAfterGetPackageInfo()
        {
            var projectParams = _report.ProjectParams;
            if (_report.GetRequiredPackageInfo(Validator.MirzaLibraryPackageName).Exists)
            {
                DefineUtility.AddDefine(DefineUtility.UsingMirzaDefine);
                if ((int)PlayerSettings.Android.minSdkVersion < 33)
                {
                    projectParams.AndroidMinimumApiLevel.Expected = "33";
                    projectParams.AndroidMinimumApiLevel.Message = "Minimum Android SDK version is not correct. Please set it to 33 or higher.";
                }
                if ((int)PlayerSettings.Android.targetSdkVersion < 33 && (int)PlayerSettings.Android.targetSdkVersion != 0)
                {
                    projectParams.AndroidTargetApiLevel.Expected = "33";
                    projectParams.AndroidTargetApiLevel.Message = "Target Android SDK version is not correct. Please set it to 33 or higher.";
                }
            }
            else
            {
                DefineUtility.RemoveDefine(DefineUtility.UsingMirzaDefine);
                if ((int)PlayerSettings.Android.minSdkVersion < 29)
                {
                    projectParams.AndroidMinimumApiLevel.Expected = "29";
                    projectParams.AndroidMinimumApiLevel.Message = "Minimum Android SDK version is not correct. Please set it to 29 or higher.";
                }
                if ((int)PlayerSettings.Android.targetSdkVersion < 31 && (int)PlayerSettings.Android.targetSdkVersion != 0)
                {
                    projectParams.AndroidTargetApiLevel.Expected = "31";
                    projectParams.AndroidTargetApiLevel.Message = "Target Android SDK version is not correct. Please set it to 31 or higher.";
                }
            }
        }

        private void CheckListRequest()
        {
            if (_listRequest is { IsCompleted: true })
            {
                foreach (var packageInfo in _listRequest.Result)
                {
                    foreach (var info in _report.ProjectParams.Packages)
                    {
                        if (packageInfo.name == info.Name)
                        {
                            info.Exists = true;
                            info.ActualVersion = packageInfo.version;
                        }
                    }
                }
                EditorApplication.update -= CheckListRequest;
                ValidateAfterGetPackageInfo();
                IsDone = true;
                _listRequest = null;
                _onComplete?.Invoke(_report);
            }
        }

        private void ValidateSamples()
        {
            var packages = UnityEditor.PackageManager.PackageInfo.GetAllRegisteredPackages().Where(package => package.packageId.StartsWith("com.qualcomm") || package.packageId.StartsWith(MirzaLibraryPackageName)).ToList();

            foreach (var package in packages)
            {
                var samples = Sample.FindByPackage(package.name, package.version).ToList();
                foreach (var sample in samples)
                {
                    var requiredAsset = new AssetParam
                    {
                        Name = sample.displayName,
                        PackageName = package.name,
                        PackageDisplayName = package.displayName,
                        Exists = sample.isImported
                    };
                    Debug.Log(requiredAsset.Name + " " + requiredAsset.Exists + " " + requiredAsset.PackageName + " " + requiredAsset.PackageDisplayName);
                    _report.ProjectParams.Assets.Add(requiredAsset);
                }
            }
        }

        private void ValidateCameras()
        {
            _report.SceneParams.Cameras = new List<CameraParam>();
            var cameras = Object.FindObjectsByType<Camera>(FindObjectsInactive.Include, FindObjectsSortMode.InstanceID);
            var xrOrigins = Object.FindObjectsByType<XROrigin>(FindObjectsInactive.Include, FindObjectsSortMode.InstanceID);

            foreach (var camera in cameras)
            {
                string targetEyeValue;
                bool isUrpPhoneCamera = false;

                if (GraphicsSettings.currentRenderPipeline == null)
                {
                    // Build-in RP
                    targetEyeValue = camera.stereoTargetEye.ToString() + "(Built-In)";
                }
                else
                {
                    // URP
                    var urpAssembly = AssemblyUtility.FindAssemblyByName("Unity.RenderPipelines.Universal.Runtime");
                    var urpAssetType = urpAssembly?.GetType("UnityEngine.Rendering.Universal.UniversalRenderPipelineAsset");

                    if (urpAssetType != null && urpAssetType.IsInstanceOfType(GraphicsSettings.currentRenderPipeline))
                    {
                        targetEyeValue = GetUrpCameraTargetEyeValue(camera);
                        if (camera.gameObject.GetComponent("SpacesHostView") != null)
                        {
                            isUrpPhoneCamera = true;
                        }
                    }
                    else
                    {
                        Debug.LogWarning("Project's RenderPipeline is neither Built-In nor URP. Not Supported.");
                        targetEyeValue = "-";
                    }
                }

                string targetDisplayValue = $"Display {camera.targetDisplay + 1}";
                string expectedDisplayValue = "";
                bool isXROriginCamera = false;
                if (_isDualRenderFusionMode)
                {
                    foreach (var xROrigin in xrOrigins)
                    {
                        if (xROrigin.Camera == camera)
                        {
                            expectedDisplayValue = "Display 2";
                            isXROriginCamera = true;
                            break;
                        }
                    }

                }

                Param targetDisplay = new Param()
                {
                    Exists = true,
                    Actual = targetDisplayValue,
                    Expected = expectedDisplayValue
                };
                var requiredCamera = new CameraParam
                {
                    TargetCamera = camera,
                    Name = camera.name,
                    ScenePath = Utility.GetHierarchyPath(camera.gameObject),
                    Tag = camera.tag,
                    Layer = camera.gameObject.layer.ToString(),
                    TargetDisplay = targetDisplay,
                    TargetEye = targetEyeValue,
                    IsActive = camera.gameObject.activeInHierarchy,
                    IsActiveSelf = camera.gameObject.activeSelf,
                    IsUrpPhoneCamera = isUrpPhoneCamera,
                    IsXrOriginCamera = isXROriginCamera
                };
                _report.SceneParams.Cameras.Add(requiredCamera);
            }
        }

        private void ValidateSimulatorParams()
        {
            var p = _report.SimulatorParams;

            p.InitializeXROnStartupPC = ValidationUtility.GetInitializeXrOnStartup(BuildTargetGroup.Standalone, _isDualRenderFusionMode);

            p.InitializeXROnStartupAndroid = ValidationUtility.GetInitializeXrOnStartup(BuildTargetGroup.Android, _isDualRenderFusionMode);

            p.SpacesXRSimulator = ValidationUtility.GetSpacesXRSimulator();

            p.StartConnected = ValidationUtility.GetStartConnected();

            p.PointerSimulator = ValidationUtility.GetPointerSimulator();

            p.PointerGameObject = ValidationUtility.GetPointerGameObject();

            p.OpenXRPluginEnabled = new Param()
            {
                Exists = true,
                Actual = Utility.IsOpenXrPluginEnabled(BuildTargetGroup.Standalone).ToString(),
                Expected = true.ToString()
            };

            p.XRSimulationPluguinEnabled = new Param()
            {
                Exists = true,
                Actual = Utility.IsXrSimulationPluginEnabled(BuildTargetGroup.Standalone).ToString(),
                Expected = true.ToString()
            };

            p.BaseRuntimeFeatureEnabled = ValidationUtility.GetBaseRuntimeFeature(BuildTargetGroup.Standalone);

            p.XRDeviceSimulatorEnabled = ValidationUtility.GetXRDeviceSimulatorEnabled();

            p.HandTrackingSimulationEnabled = ValidationUtility.GetHandTrackingSimulationEnabled();

            p.MirzaDeviceEventSimulator = ValidationUtility.GetMirzaDeviceEventSimulator();
        }

        private void ValidateTemplatesParams()
        {
            var p = _report.TemplatesParams;

            p.HandTrackingFeatureEnabled = ValidationUtility.GetHandTrackingFeature();
        }

        private void ValidateSamplesParams()
        {
            var p = _report.SamplesParams;

            p.CameraAccessFeatureEnabled = ValidationUtility.GetCameraAccessFeature();

            p.AllowUnsafeCode = ValidationUtility.GetAllowUnsafeCode();

            p.OpenAIAPIKeyConfigured = ValidationUtility.GetOpenAIAPIKeyConfigured();

            p.GCPAPIKeyConfigured = ValidationUtility.GetGCPAPIKeyConfigured();
        }

        private void ValidateDebugParams()
        {
            var p = _report.DebugParams;

            p.FrameTimingStatsEnabled = ValidationUtility.GetFrameTimingStatsEnabled();
            p.DebugPrefabAdded = ValidationUtility.DebugPrefabAdded();
        }

        private string GetUrpCameraTargetEyeValue(Camera target)
        {
            string assemblyName = "Unity.RenderPipelines.Universal.Runtime";
            string methodName = "GetUniversalAdditionalCameraData";

            var targetAssembly = AssemblyUtility.FindAssemblyByName(assemblyName);

            if (targetAssembly == null)
            {
                Debug.LogError("Assembly not found: " + assemblyName);
                return "-";
            }

            MethodInfo getUniversalAdditionalCameraDataMethod = null;
            var types = targetAssembly.GetTypes();
            foreach (var type in types)
            {
                if (!type.IsSealed || !type.IsAbstract) // Only static classes
                    continue;

                var methods = type.GetMethods(BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic);
                foreach (var method in methods)
                {
                    // Extensionメソッド限定、メソッド名検索、パラメータ情報からGetUniversalAdditionalCameraDataメソッドを特定する
                    if (method.IsDefined(typeof(System.Runtime.CompilerServices.ExtensionAttribute), false) &&
                        method.Name == methodName &&
                        method.GetParameters().Length > 0 &&
                        method.GetParameters()[0].ParameterType == typeof(Camera)) // Check for Camera type
                    {
                        getUniversalAdditionalCameraDataMethod = method;
                        break;
                    }
                }

                if (getUniversalAdditionalCameraDataMethod != null)
                    break;
            }

            // GetUniversalAdditionalCameraData()の戻り値を保存
            // 型はUnityEngine.Rendering.Universal.UniversalAdditionalCameraData
            var cameraData = getUniversalAdditionalCameraDataMethod.Invoke(null, new object[] { target });
            if (cameraData == null)
            {
                Debug.LogError("Could not retrieve UniversalAdditionalCameraData instance.");
                return "-";
            }


            // UniversalAdditionalCameraDataクラスからallowXRRenderingプロパティを取得する
            string universalAdditionalCameraDataName = "UnityEngine.Rendering.Universal.UniversalAdditionalCameraData";

            var universalAdditionalCameraDataType = targetAssembly.GetType(universalAdditionalCameraDataName);
            if (universalAdditionalCameraDataType == null)
            {
                Debug.LogError("Type not found: " + universalAdditionalCameraDataName);
                return "-";
            }
            var allowXRRenderingProperty = universalAdditionalCameraDataType.GetProperty("allowXRRendering", BindingFlags.Instance | BindingFlags.Public);

            if (allowXRRenderingProperty == null)
            {
                Debug.LogError("allowXRRendering property not found.");
                return "-";
            }

            // UniversalAdditionalCameraData型のインスタンスからallowXRRenderingプロパティの値を取得する
            bool currentValue = (bool)allowXRRenderingProperty.GetValue(cameraData);

            return currentValue ? "Both(URP)" : Constants.TargetEyeIsNoneInUrpProject;
        }

        private Version GetUnityVersion()
        {
            var version = Application.unityVersion;
            var versionParts = version.Split('.', 'f');
            if (versionParts.Length != 4)
            {
                return null;
            }
            else
            {
                return new Version(int.Parse(versionParts[0]), int.Parse(versionParts[1]), int.Parse(versionParts[2]), int.Parse(versionParts[3]));
            }
        }
    }
}