﻿using System;
using System.Linq;
using System.IO;
using Unity.XR.CoreUtils;
using UnityEditor;
using UnityEditor.XR.Management;
using UnityEditor.XR.OpenXR.Features;
using UnityEngine;
using UnityEngine.XR.ARFoundation;
using UnityEngine.XR.OpenXR;
using Object = UnityEngine.Object;
using NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Runtime;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public static class ValidationUtility
    {
        #region ProjectParams

        public static Param GetSpacesFeatureEnabled()
        {
            var fs = OpenXRFeatureSetManager.GetFeatureSetWithId(BuildTargetGroup.Android,
                "com.qualcomm.snapdragon.spaces");
            if (fs == null)
            {
                return Param.CreateNotExists();
            }
            var vi = new Param
            {
                Exists = true,
                Expected = true.ToString(),
                Actual = fs.isEnabled.ToString()
            };
            return vi;
        }

        public static Param GetSpacesExperimentalFeatureEnabled()
        {
            var fs = OpenXRFeatureSetManager.GetFeatureSetWithId(BuildTargetGroup.Android,
                "com.qualcomm.snapdragon.spaces.experimental");
            if (fs == null)
            {
                return Param.CreateNotExists();
            }
            var vi = new Param
            {
                Exists = true,
                Expected = true.ToString(),
                Actual = fs.isEnabled.ToString()
            };
            return vi;
        }

        public static Param GetInitializeXrOnStartup(BuildTargetGroup targetGroup, bool isDualRenderFusionMode)
        {
            var xrGeneralSettings = XRGeneralSettingsPerBuildTarget.XRGeneralSettingsForBuildTarget(targetGroup);
            if (xrGeneralSettings == null)
            {
                return Param.CreateNotExists();
            }

            if (isDualRenderFusionMode)
            {
                var vi = new Param
                {
                    Exists = true,
                    Actual = xrGeneralSettings.InitManagerOnStart.ToString(),
                    Expected = false.ToString(),
                    Message = "This setting should be set to False for Dual Render Fusion mode."
                };
                return vi;
            }
            else
            {
                var vi = new Param
                {
                    Exists = true,
                    Actual = xrGeneralSettings.InitManagerOnStart.ToString(),
                    Expected = true.ToString(),
                    Message = "This setting should be set to True for Non-Dual Render Fusion mode."
                };
                return vi;
            }
        }

        public static Param GetActiveInputHandling(bool isDualRenderFusionMode)
        {
            var settings = Resources.FindObjectsOfTypeAll<PlayerSettings>().FirstOrDefault();
            if (settings == null)
            {
                return Param.CreateNotExists();
            }

            var settingsObject = new SerializedObject(settings);
            var property = settingsObject.FindProperty("activeInputHandler");
            if (property == null)
            {
                return Param.CreateNotExists();
            }

            string valueString = null;
            switch (property.intValue)
            {
                case 0:
                    valueString = "Input Manager (Old)";
                    break;
                case 1:
                    valueString = "Input System Package (New)";
                    break;
                case 2:
                    valueString = "Both";
                    break;
            }

            if (isDualRenderFusionMode)
            {
                var vi = new Param
                {
                    Exists = true,
                    Actual = valueString,
                    Expected = "Both",
                    Message = "This setting should be set to Both for Dual Render Fusion mode."
                };
                return vi;
            }
            else
            {
                var vi = new Param
                {
                    Exists = true,
                    Actual = valueString
                };
                return vi;
            }
        }

        public static Param GetAllowUnsafeCode()
        {
            var settings = Resources.FindObjectsOfTypeAll<PlayerSettings>().FirstOrDefault();
            if (settings == null)
            {
                return Param.CreateNotExists();
            }

            var settingsObject = new SerializedObject(settings);
            var property = settingsObject.FindProperty("allowUnsafeCode");
            if (property == null)
            {
                return Param.CreateNotExists();
            }

            string valueString = property.boolValue.ToString();
            var vi = new Param
            {
                Exists = true,
                Actual = valueString
            };
            return vi;
        }

        public static Param GetLaunchAppOnViewer(bool isDualRenderFusionMode)
        {
            var succeed = SpacesAssemblyUtility.TryGetLaunchAppOnViewer(out var result);
            if (!succeed)
            {
                return Param.CreateNotExists();
            }

            if (isDualRenderFusionMode)
            {
                var vi = new Param
                {
                    Exists = true,
                    Expected = false.ToString(),
                    Actual = result.ToString(),
                    Message = "This setting should be set to False for Dual Render Fusion mode."
                };
                return vi;
            }
            else
            {
                var vi = new Param
                {
                    Exists = true,
                    Expected = true.ToString(),
                    Actual = result.ToString(),
                    Message = "This setting should be set to True for Non-Dual Render Fusion mode."
                };
                return vi;
            }
        }

        public static Param GetLaunchControllerOnHost(bool isDualRenderFusionMode)
        {
            var succeed = SpacesAssemblyUtility.TryGetLaunchControllerOnHost(out var result);
            if (!succeed)
            {
                return Param.CreateNotExists();
            }

            if (isDualRenderFusionMode)
            {
                var vi = new Param
                {
                    Exists = true,
                    Expected = false.ToString(),
                    Actual = result.ToString(),
                    Message = "This setting should be set to False for Dual Render Fusion mode."
                };
                return vi;
            }
            else
            {
                var vi = new Param
                {
                    Exists = true,
                    Expected = true.ToString(),
                    Actual = result.ToString(),
                    Message = "This setting should be set to True for Non-Dual Render Fusion mode."
                };
                return vi;
            }
        }

        public static Param GetExportHeadless(bool isDualRenderFusionMode)
        {
            var succeed = SpacesAssemblyUtility.TryGetExportHeadless(out var result);
            if (!succeed)
            {
                return Param.CreateNotExists();
            }

            if (isDualRenderFusionMode)
            {
                var vi = new Param
                {
                    Exists = true,
                    Expected = false.ToString(),
                    Actual = result.ToString(),
                    Message = "This setting should be set to False for Dual Render Fusion mode."
                };
                return vi;
            }
            else
            {
                var vi = new Param
                {
                    Exists = true,
                    Expected = true.ToString(),
                    Actual = result.ToString(),
                    Message = "This setting should be set to True for Non-Dual Render Fusion mode."
                };
                return vi;
            }
        }

        public static Param GetBaseRuntimeFeature(BuildTargetGroup targetGroup)
        {
            string targetName;
            switch (targetGroup)
            {
                case BuildTargetGroup.Android:
                    targetName = "Android";
                    break;
                default:
                    targetName = "Standalone";
                    break;
            }

            var settings = Resources.FindObjectsOfTypeAll<OpenXRSettings>().FirstOrDefault(s => s.name == targetName);
            if (settings == null)
            {
                return Param.CreateNotExists();
            }

            var feature = settings.GetFeatures().FirstOrDefault(f => f.name == $"BaseRuntimeFeature {targetGroup}");
            if (feature == null)
            {
                return Param.CreateNotExists();
            }

            var vi = new Param
            {
                Exists = true,
                Expected = true.ToString(),
                Actual = feature.enabled.ToString(),
            };
            return vi;
        }

        public static Param GetHandTrackingFeature()
        {
            var settings = Resources.FindObjectsOfTypeAll<OpenXRSettings>().FirstOrDefault(s => s.name == "Android");
            if (settings == null)
            {
                return Param.CreateNotExists();
            }

            try
            {
                var feature = settings.GetFeatures().FirstOrDefault(f => f.name == "HandTrackingFeature Android");
                if (feature == null)
                {
                    return Param.CreateNotExists();
                }

                var vi = new Param
                {
                    Exists = true,
                    Expected = true.ToString(),
                    Actual = feature.enabled.ToString(),
                };
                return vi;
            }
            catch (Exception)
            {
                return Param.CreateNotExists();
            }
        }

        public static Param GetCameraAccessFeature()
        {
            var settings = Resources.FindObjectsOfTypeAll<OpenXRSettings>().FirstOrDefault(s => s.name == "Android");
            if (settings == null)
            {
                return Param.CreateNotExists();
            }

            try
            {
                var feature = settings.GetFeatures().FirstOrDefault(f => f.name == "CameraAccessFeature Android");
                if (feature == null)
                {
                    return Param.CreateNotExists();
                }

                var vi = new Param
                {
                    Exists = true,
                    Expected = true.ToString(),
                    Actual = feature.enabled.ToString(),
                };
                return vi;
            }
            catch (Exception)
            {
                return Param.CreateNotExists();
            }
        }

        public static Param GetDualRenderFusionFeature()
        {
            var settings = Resources.FindObjectsOfTypeAll<OpenXRSettings>().FirstOrDefault(s => s.name == "Android");
            if (settings == null)
            {
                return Param.CreateNotExists();
            }

            var feature = settings.GetFeatures().FirstOrDefault(f => f.name == "FusionFeature Android");
            if (feature == null)
            {
                return Param.CreateNotExists();
            }

            var vi = new Param
            {
                Exists = true,
                Expected = true.ToString(),
                Actual = feature.enabled.ToString(),
                Message = "This setting should be set to True for Dual Render Fusion mode."
            };
            return vi;
        }

        public static Param GetIgnoreOpenXRVersion()
        {
            var succeed = SpacesAssemblyUtility.TryGetIgnoreOpenXRVersion(out var result);
            if (!succeed)
            {
                return Param.CreateNotExists();
            }

            var vi = new Param
            {
                Exists = true,
                Actual = result.ToString(),
            };
            return vi;
        }

        public static Param GetXRDeviceSimulatorEnabled()
        {
            var succeed = XRInteractionToolkitAssemblyUtility.TryGetXRDeviceSimulatorEnabled(out var result);
            if (!succeed)
            {
                return Param.CreateNotExists();
            }

            var vi = new Param
            {
                Exists = true,
                Expected = true.ToString(),
                Actual = result.ToString(),
            };
            return vi;
        }

        public static Param GetHandTrackingSimulationEnabled()
        {
            var succeed = QCHTAssemblyUtility.TryGetHandTrackingSimulationEnabled(out var result);
            if (!succeed)
            {
                return Param.CreateNotExists();
            }

            var vi = new Param
            {
                Exists = true,
                Expected = true.ToString(),
                Actual = result.ToString(),
            };
            return vi;
        }

        #endregion

        #region SceneParams

        public static Param GetARSession()
        {
            var arSession = Object.FindFirstObjectByType<ARSession>(FindObjectsInactive.Include);
            if (arSession == null)
            {
                return Param.CreateNotExists();
            }

            return new Param
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(arSession.gameObject),
                Message = "ARSession should be in the scene."
            };
        }

        public static Param GetXROrigin()
        {
            var xrOrigin = Object.FindFirstObjectByType<XROrigin>(FindObjectsInactive.Include);
            if (xrOrigin == null)
            {
                return Param.CreateNotExists();
            }

            return new Param
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(xrOrigin.gameObject),
                Message = "XROrigin should be in the scene."
            };
        }

        public static Param GetARCameraManager()
        {
            var arCameraManager = Object.FindFirstObjectByType<ARCameraManager>(FindObjectsInactive.Include);
            if (arCameraManager == null)
            {
                return Param.CreateNotExists();
            }

            return new Param
            {
                Exists = true,
                Actual =
                    $"{Utility.GetHierarchyPath(arCameraManager.gameObject)} ({(arCameraManager.enabled ? "Enabled" : "Disabled")})",
                Message = "ARCameraManager should be disabled."
            };
        }

        public static Param GetARCameraBackground()
        {
            var arCameraBackground = Object.FindFirstObjectByType<ARCameraBackground>(FindObjectsInactive.Include);
            if (arCameraBackground == null)
            {
                return Param.CreateNotExists();
            }

            return new Param
            {
                Exists = true,
                Actual =
                    $"{Utility.GetHierarchyPath(arCameraBackground.gameObject)} ({(arCameraBackground.enabled ? "Enabled" : "Disabled")})",
                Message = "ARCameraBackground should be disabled."
            };
        }

        public static Param GetDynamicOpenXRLoader()
        {
            var target = Utility.FindFirstComponentByName("DynamicOpenXRLoader");
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject),
                Message = "DynamicOpenXRLoader is recommended to use Dual Render Fusion mode."
            };
        }

        public static Param GetSpacesHostView()
        {
            var target = Utility.FindFirstComponentByName("SpacesHostView");
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject),
                Message = "SpacesHostView is required to use Dual Render Fusion mode."
            };
        }

        public static Param GetSpacesXRSimulator()
        {
            var target = Utility.FindFirstComponentByName("SpacesXRSimulator");
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject)
            };
        }

        public static Param GetStartConnected()
        {
            var target = Utility.FindFirstComponentByName("SpacesXRSimulator");
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            if (SpacesAssemblyUtility.TryGetStartConnected(out var result))
            {
                return new Param()
                {
                    Exists = true,
                    Actual = result.ToString(),
                    Expected = true.ToString(),
                };
            }
            else
            {
                return Param.CreateNotExists();
            }
        }

        public static Param GetPointerSimulator()
        {
            var target = Utility.FindFirstComponentByName("PointerSimulator");
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject)
            };
        }

        public static Param GetPointerGameObject()
        {
            var obj = Utility.FindFirstComponentByName("PointerSimulator");
            if (obj == null)
            {
                return Param.CreateNotExists();
            }

            var simulator = obj as PointerSimulator;
            var pointerGameObject = simulator.Pointer;
            if (pointerGameObject == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(pointerGameObject)
            };
        }

        public static Param GetDualRenderFusionController()
        {
            var target = Utility.FindGameObjectByName(Constants.ControllerGameObjectName);
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target)
            };
        }

        public static Param GetRenderCamera()
        {
            var target = Utility.FindGameObjectByName(Constants.RenderCameraGameObjectName);
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target)
            };
        }

        public static Param GetXRInteractionManager()
        {
            var target = Utility.FindFirstComponentByName("XRInteractionManager");
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject)
            };
        }

        public static Param GetConfiguredInputActionManager()
        {
            if (XRInteractionToolkitAssemblyUtility.TryGetConfiguredInputActionManager(out var result))
            {
                return new Param()
                {
                    Exists = true,
                    Actual = Utility.GetHierarchyPath(result.gameObject),
                };
            }
            else
            {
                return Param.CreateNotExists();
            }

        }

        public static Param GetQCHTXROrigin()
        {
            var target = Utility.FindGameObjectByName(Constants.QCHTXROriginGameObjectName);
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target)
            };
        }

        public static Param GetMirzaDeviceEventSimulator()
        {
            var target = Utility.FindFirstComponentByName("MirzaDeviceEventSimulator");
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject)
            };
        }

        public static Param GetSpacesPermissionHelper()
        {
            var target = Utility.FindFirstComponentByName("SpacesPermissionHelper");
            if (target == null)
            {
                return Param.CreateNotExists();
            }

            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject),
                Message = "SpacesPermissionHelper is required to setup for request camera permission on start."
            };
        }

        public static Param GetSpacesLifecycleEvents()
        {
            var target = Utility.FindFirstComponentByName("SpacesLifecycleEvents");
            if (target == null)
            {
                return Param.CreateNotExists();
            }
            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject),
                Message = "SpacesLifecycleEvents is required to Spaces Life Cycle Event Configuration and Setup For Request Camera Permission."
            };
        }

        public static Param GetMRModeAutoSwitcher()
        {
            var target = Utility.FindGameObjectByName("MR Mode Auto Switcher");
            if (target == null)
            {
                return Param.CreateNotExists();
            }
            return new Param()
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(target.gameObject),
                Message = "MRModeAutoSwitcher is required to switch to MR Mode without MiRZA App."
            };
        }

        #endregion

        #region SamplesParams

        public static Param GetOpenAIAPIKeyConfigured()
        {
            var gptImageAnalyzer = Utility.FindFirstComponentByName("GPTImageAnalyzer");
            if (gptImageAnalyzer == null)
            {
                return Param.CreateNotExists();
            }

            var canGetOpenAIAPIKeyText = AssemblyUtility.TryGetFieldValueFromAssembly("Assembly-CSharp", "NTTQONOQ.Android.MiRZA.Library.Samples.GPTImageAnalyzerSample.GPTImageAnalyzer", "_openAiApiKeyText", gptImageAnalyzer, out var openAIAPIKeyTextObj);
            var canGetOpenAIAPIKey = AssemblyUtility.TryGetFieldValueFromAssembly("Assembly-CSharp", "NTTQONOQ.Android.MiRZA.Library.Samples.GPTImageAnalyzerSample.GPTImageAnalyzer", "_openAiApiKey", gptImageAnalyzer, out var openAIAPIKeyObj);

            if (!canGetOpenAIAPIKeyText || !canGetOpenAIAPIKey)
            {
                return Param.CreateNotExists();
            }

            var openAIAPIKeyText = openAIAPIKeyTextObj as TextAsset;
            var openAIAPIKey = openAIAPIKeyObj as string;
            if (openAIAPIKeyText == null && string.IsNullOrEmpty(openAIAPIKey))
            {
                return new Param()
                {
                    Exists = true,
                    Actual = false.ToString(),
                    Expected = true.ToString(),
                };
            }
            else
            {
                return new Param()
                {
                    Exists = true,
                    Actual = true.ToString(),
                    Expected = true.ToString(),
                };
            }
        }

        public static Param GetGCPAPIKeyConfigured()
        {
            var googleTextToSpeechConverter = Utility.FindFirstComponentByName("GoogleTextToSpeechConverter");
            if (googleTextToSpeechConverter == null)
            {
                return Param.CreateNotExists();
            }

            var canGetGcpAPIKeyText = AssemblyUtility.TryGetFieldValueFromAssembly("Assembly-CSharp", "NTTQONOQ.Android.MiRZA.Library.Samples.GPTImageAnalyzerSample.GoogleTextToSpeechConverter", "_gcpApiKeyText", googleTextToSpeechConverter, out var gcpAPIKeyTextObj);
            var canGetGcpAPIKey = AssemblyUtility.TryGetFieldValueFromAssembly("Assembly-CSharp", "NTTQONOQ.Android.MiRZA.Library.Samples.GPTImageAnalyzerSample.GoogleTextToSpeechConverter", "_gcpApiKey", googleTextToSpeechConverter, out var gcpAPIKeyObj);

            if (!canGetGcpAPIKeyText || !canGetGcpAPIKey)
            {
                return Param.CreateNotExists();
            }

            var gcpAPIKeyText = gcpAPIKeyTextObj as TextAsset;
            var gcpAPIKey = gcpAPIKeyObj as string;

            if (gcpAPIKeyText == null && string.IsNullOrEmpty(gcpAPIKey))
            {
                return new Param()
                {
                    Exists = true,
                    Actual = false.ToString(),
                    Expected = true.ToString(),
                };
            }
            else
            {
                return new Param()
                {
                    Exists = true,
                    Actual = true.ToString(),
                    Expected = true.ToString(),
                };
            }
        }

        [Serializable]
        private class PackageJson
        {
            public string version;
        }

        #endregion

        #region DebugParams

        public static Param GetFrameTimingStatsEnabled()
        {
            var vi = new Param
            {
                Exists = true,
                Actual = PlayerSettings.enableFrameTimingStats.ToString()
            };
            return vi;
        }

        public static Param DebugPrefabAdded()
        {
            var prefab = GameObject.Find("IngameDebugConsoleForMirza");
            if (prefab == null)
            {
                return Param.CreateNotExists();
            }

            return new Param
            {
                Exists = true,
                Actual = Utility.GetHierarchyPath(prefab.gameObject)
            };
        }

        #endregion
    }
}