﻿using System.Collections.Generic;
using System.IO;
using System.Linq;
using Unity.XR.CoreUtils;
using UnityEditor;
using UnityEditor.XR.Management;
using UnityEditor.XR.Management.Metadata;
using UnityEngine;
using UnityEngine.SceneManagement;
using UnityEngine.UIElements;
using Object = UnityEngine.Object;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public static class Utility
    {
        public static bool IsOpenXrPluginEnabled(BuildTargetGroup buildTargetGroup)
        {
            var xrGeneralSettings = XRGeneralSettingsPerBuildTarget.XRGeneralSettingsForBuildTarget(buildTargetGroup);
            if (xrGeneralSettings == null)
            {
                return false;
            }

            var xrManagerSettings = xrGeneralSettings.AssignedSettings;
            if (xrManagerSettings == null)
            {
                return false;
            }

            if (xrManagerSettings.activeLoaders.Any(loader => loader.GetType() == typeof(UnityEngine.XR.OpenXR.OpenXRLoader)))
            {
                return true;
            }

            return false;
        }

        public static void SetOpenXrPluginEnabled(BuildTargetGroup buildTargetGroup, bool enabled)
        {
            var xrGeneralSettings = XRGeneralSettingsPerBuildTarget.XRGeneralSettingsForBuildTarget(buildTargetGroup);
            if (xrGeneralSettings == null)
            {
                throw new System.Exception("XRGeneralSettings is not found.");
            }

            var xrManagerSettings = xrGeneralSettings.AssignedSettings;
            if (xrManagerSettings == null)
            {
                throw new System.Exception("AssignedSettings is not found.");
            }

            if (enabled)
            {
                XRPackageMetadataStore.AssignLoader(xrManagerSettings, typeof(UnityEngine.XR.OpenXR.OpenXRLoader).FullName, buildTargetGroup);
            }
            else
            {
                XRPackageMetadataStore.RemoveLoader(xrManagerSettings, typeof(UnityEngine.XR.OpenXR.OpenXRLoader).FullName, buildTargetGroup);
            }
        }

        public static bool IsXrSimulationPluginEnabled(BuildTargetGroup buildTargetGroup)
        {
            var xrGeneralSettings = XRGeneralSettingsPerBuildTarget.XRGeneralSettingsForBuildTarget(buildTargetGroup);
            if (xrGeneralSettings == null)
            {
                return false;
            }

            var xrManagerSettings = xrGeneralSettings.AssignedSettings;
            if (xrManagerSettings == null)
            {
                return false;
            }

            if (xrManagerSettings.activeLoaders.Any(loader => loader.GetType() == typeof(UnityEngine.XR.Simulation.SimulationLoader)))
            {
                return true;
            }

            return false;
        }

        public static void SetXrSimulationPluginEnabled(BuildTargetGroup buildTargetGroup, bool enabled)
        {
            var xrGeneralSettings = XRGeneralSettingsPerBuildTarget.XRGeneralSettingsForBuildTarget(buildTargetGroup);
            if (xrGeneralSettings == null)
            {
                throw new System.Exception("XRGeneralSettings is not found.");
            }

            var xrManagerSettings = xrGeneralSettings.AssignedSettings;
            if (xrManagerSettings == null)
            {
                throw new System.Exception("AssignedSettings is not found.");
            }

            if (enabled)
            {
                XRPackageMetadataStore.AssignLoader(xrManagerSettings, typeof(UnityEngine.XR.Simulation.SimulationLoader).FullName, buildTargetGroup);
            }
            else
            {
                XRPackageMetadataStore.RemoveLoader(xrManagerSettings, typeof(UnityEngine.XR.Simulation.SimulationLoader).FullName, buildTargetGroup);
            }
        }

        public static string GetHierarchyPath(GameObject obj)
        {
            var path = obj.name;
            var parent = obj.transform.parent;
            while (parent != null)
            {
                path = parent.name + "/" + path;
                parent = parent.parent;
            }
            return "/" + path;
        }

        public static GameObject FindGameObjectByName(string name)
        {
            var rootObjects = Object.FindObjectsByType<GameObject>(FindObjectsInactive.Include, FindObjectsSortMode.None);
            var target = rootObjects.FirstOrDefault(o => o.name == name);
            return target;
        }

        public static GameObject FindGameObjectByHierarchyPath(string hierarchyPath)
        {
            var rootObjects = Object.FindObjectsByType<GameObject>(FindObjectsInactive.Include, FindObjectsSortMode.None);
            var target = rootObjects.FirstOrDefault(o => GetHierarchyPath(o) == hierarchyPath);
            return target;
        }

        public static GameObject FindGameObjectByHierarchyPathEndsWith(string hierarchyPath)
        {
            var rootObjects = Object.FindObjectsByType<GameObject>(FindObjectsInactive.Include, FindObjectsSortMode.None);
            var target = rootObjects.FirstOrDefault(o => GetHierarchyPath(o).EndsWith(hierarchyPath));
            return target;
        }

        public static GameObject[] FindGameObjectsByTag(string tag)
        {
            var rootObjects = Object.FindObjectsByType<GameObject>(FindObjectsInactive.Include, FindObjectsSortMode.None);
            var targets = rootObjects.Where(o => o.CompareTag(tag)).ToArray();
            return targets;
        }

        public static GameObject FindGameObjectByTag(string tag)
        {
            var rootObjects = Object.FindObjectsByType<GameObject>(FindObjectsInactive.Include, FindObjectsSortMode.None);
            var target = rootObjects.FirstOrDefault(o => o.CompareTag(tag));
            return target;
        }

        public static MonoBehaviour FindFirstComponentByName(string componentName)
        {
            var objects = Object.FindObjectsByType<MonoBehaviour>(FindObjectsInactive.Include, FindObjectsSortMode.None);
            var target = objects.FirstOrDefault(o => o.GetType().Name == componentName);
            return target;
        }

        public static List<MonoBehaviour> FindComponentsByName(string componentName)
        {
            var objects = Object.FindObjectsByType<MonoBehaviour>(FindObjectsInactive.Include, FindObjectsSortMode.None);
            var targets = objects.Where(o => o.GetType().Name == componentName).ToList();
            return targets;
        }

        public static MonoBehaviour FindAttachedComponentByName(GameObject target, string componentName)
        {
            var components = target.GetComponents<MonoBehaviour>();
            var component = components.FirstOrDefault(c => c.GetType().Name == componentName);
            return component;
        }

        public static Transform FindChildObjectRecursive(Transform parent, string name)
        {
            foreach (Transform child in parent)
            {
                if (child.name == name)
                {
                    return child;
                }
                var result = FindChildObjectRecursive(child, name);
                if (result != null)
                {
                    return result;
                }
            }
            return null;
        }

        /// <summary>
        /// タイプ指定でシーン内の全てのオブジェクトを取得する
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <returns></returns>
        public static List<T> FindComponentsByType<T>() where T : Component
        {
            List<T> results = new List<T>();
            // オープンしているシーンの数を取得
            int sceneCount = SceneManager.sceneCount;

            for (int i = 0; i < sceneCount; i++)
            {
                Scene scene = SceneManager.GetSceneAt(i);
                GameObject[] rootObjects = scene.GetRootGameObjects();
                foreach (GameObject rootObject in rootObjects)
                {
                    T[] componentsInChildren = rootObject.GetComponentsInChildren<T>(true);
                    results.AddRange(componentsInChildren);
                }
            }
            return results;
        }

        public static void SetInitializeXrOnStartup(BuildTargetGroup targetGroup, bool enabled)
        {
            var xrGeneralSettings = XRGeneralSettingsPerBuildTarget.XRGeneralSettingsForBuildTarget(targetGroup);
            if (xrGeneralSettings == null)
            {
                throw new System.Exception("XRGeneralSettings is not found.");
            }
            xrGeneralSettings.InitManagerOnStart = enabled;
        }

        public static void SetBaseRuntimeFeatureEnabled(BuildTargetGroup targetGroup, bool enabled)
        {
            string targetName;
            switch (targetGroup)
            {
                case BuildTargetGroup.Android:
                    targetName = "Android";
                    break;
                default:
                    targetName = "Standalone";
                    break;
            }

            var settings = Resources.FindObjectsOfTypeAll<UnityEngine.XR.OpenXR.OpenXRSettings>().FirstOrDefault(s => s.name == targetName);
            if (settings == null)
            {
                throw new System.Exception("OpenXRSettings is not found.");
            }

            var feature = settings.GetFeatures().FirstOrDefault(f => f.name == $"BaseRuntimeFeature {targetGroup}");
            if (feature == null)
            {
                throw new System.Exception("BaseRuntimeFeature is not found.");
            }

            feature.enabled = enabled;
        }

        public static void SetAllowUnsafeCode(bool value)
        {
            var settings = Resources.FindObjectsOfTypeAll<PlayerSettings>().FirstOrDefault();
            if (settings == null)
            {
                throw new System.Exception("PlayerSettings is not found.");
            }

            var settingsObject = new SerializedObject(settings);
            var property = settingsObject.FindProperty("allowUnsafeCode");
            if (property == null)
            {
                throw new System.Exception("allowUnsafeCode is not found.");
            }

            property.boolValue = value;
            settingsObject.ApplyModifiedProperties();
        }

        public static bool IsQnqControllerFolderExist()
        {
            if (!Directory.Exists(Constants.MirzaConfigurationToolSamplesBasePath))
            {
                return false;
            }

            string[] directories = Directory.GetDirectories(Constants.MirzaConfigurationToolSamplesBasePath, "*",
                SearchOption.AllDirectories);
            foreach (string dir in directories)
            {
                string potentialPath = Path.Combine(dir, Constants.QONOQControllerFolderSearchPattern);
                if (Directory.Exists(potentialPath))
                {
                    return true;
                }
            }
            return false;
        }

        public static bool TryAddDualRenderFusionControllerPrefabToScene()
        {
            // Check if the prefab exists
            var target = FindGameObjectByName(Constants.ControllerGameObjectName);
            if (target != null)
            {
                EditorUtility.DisplayDialog("Error",
                    $"{Constants.ControllerGameObjectName} already exists in the scene.", "OK");
                Selection.activeGameObject = target;
                return false;
            }

            var prefabPath = AssetDatabase.GUIDToAssetPath(Constants.ControllerPrefabGuid);
            var prefab = AssetDatabase.LoadAssetAtPath<GameObject>(prefabPath);

            if (prefab == null)
            {
                EditorUtility.DisplayDialog("Error",
                    $"Prefab not found.\nImport the dual render fusion controller from the configuration window.",
                    "OK");
                return false;
            }

            var obj = PrefabUtility.InstantiatePrefab(prefab);
            if (obj == null)
            {
                EditorUtility.DisplayDialog("Error",
                    $"Failed to instantiate {Constants.ControllerGameObjectName} prefab.", "OK");
                return false;
            }
            Undo.RegisterCreatedObjectUndo(obj, "Add Dual Render Fusion Controller");
            Selection.activeGameObject = (GameObject)obj;

            return true;
        }

        public static bool TryAddRenderCameraPrefabToScene()
        {
            // Check if the prefab exists
            var target = FindGameObjectByName(Constants.RenderCameraGameObjectName);
            if (target != null)
            {
                EditorUtility.DisplayDialog("Error",
                    $"{Constants.RenderCameraGameObjectName} already exists in the scene.", "OK");
                Selection.activeGameObject = target;
                return false;
            }

            XROrigin xrOrigin = null;
            foreach (GameObject root in UnityEngine.SceneManagement.SceneManager.GetActiveScene().GetRootGameObjects())
            {
                xrOrigin = root.GetComponentInChildren<XROrigin>(true);
                if (xrOrigin != null)
                    break;
            }
            if (xrOrigin == null)
            {
                EditorUtility.DisplayDialog("Error",
                    "XR Origin not found in the scene.\nAdd the XR Origin to the scene first.", "OK");
                return false;
            }
            var targetCamera = xrOrigin.Camera;
            if (targetCamera == null)
            {
                EditorUtility.DisplayDialog("Error", "Camera not found in the XR Origin.", "OK");
                return false;
            }

            var prefabPath = AssetDatabase.GUIDToAssetPath(Constants.RenderCameraPrefabGuid);
            var prefab = AssetDatabase.LoadAssetAtPath<GameObject>(prefabPath);

            if (prefab == null)
            {
                EditorUtility.DisplayDialog("Error",
                    $"Prefab not found.\nImport the dual render fusion controller from the configuration window.",
                    "OK");
                return false;
            }

            var obj = PrefabUtility.InstantiatePrefab(prefab);
            var gameObj = obj as GameObject;
            gameObj.transform.parent = targetCamera.transform;
            gameObj.transform.localPosition = Vector3.zero;
            gameObj.transform.localRotation = Quaternion.identity;

            Undo.RegisterCreatedObjectUndo(obj, "Add Render Camera");
            Selection.activeGameObject = obj as GameObject;
            return true;
        }

        public static void OpenSamplesTabInPackageManagerWindow()
        {
            var packageManagerWindow = Resources.FindObjectsOfTypeAll<EditorWindow>()
                .FirstOrDefault(w => w.GetType().Name == "PackageManagerWindow");
            if (packageManagerWindow != null)
            {
                var rootVisualElement = packageManagerWindow.rootVisualElement;
                var samplesButton = rootVisualElement.Q<Button>("samplesButton");
                if (samplesButton != null)
                {
                    using (var submitEvent = NavigationSubmitEvent.GetPooled())
                    {
                        submitEvent.target = samplesButton;
                        samplesButton.SendEvent(submitEvent);
                    }
                }
            }
        }
    }
}
