﻿using System;
using System.Linq;
using UnityEditor;
using UnityEngine;
using UnityEngine.Events;
using UnityEngine.XR.OpenXR;
using Object = UnityEngine.Object;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public static class SpacesAssemblyUtility
    {
        public static void ShowSpacesSetupWindow()
        {
            AssemblyUtility.CallMethodFromAssembly(
                "Snapdragon.Spaces.Editor",
                "Qualcomm.Snapdragon.Spaces.Editor.SpacesSetup",
                "ShowSpacesSetupWindow");
        }

        public static bool TryGetLaunchAppOnViewer(out bool result)
        {
            result = false;

            var settings = Resources.FindObjectsOfTypeAll<OpenXRSettings>().FirstOrDefault(s => s.name == "Android");
            if (settings == null)
            {
                return false;
            }

            var feature = settings.GetFeatures().FirstOrDefault(f => f.name == "BaseRuntimeFeature Android");
            if (feature == null)
            {
                return false;
            }

            var succeed = AssemblyUtility.TryGetFieldValueFromAssembly("Snapdragon.Spaces.Runtime", "Qualcomm.Snapdragon.Spaces.BaseRuntimeFeature", "LaunchAppOnViewer", feature, out var field);
            if (!succeed)
            {
                return false;
            }

            if (field.GetType() != typeof(bool))
            {
                return false;
            }

            result = (bool)field;
            return true;
        }

        public static bool TryGetLaunchControllerOnHost(out bool result)
        {
            result = false;

            var settings = Resources.FindObjectsOfTypeAll<OpenXRSettings>().FirstOrDefault(s => s.name == "Android");
            if (settings == null)
            {
                return false;
            }

            var feature = settings.GetFeatures().FirstOrDefault(f => f.name == "BaseRuntimeFeature Android");
            if (feature == null)
            {
                return false;
            }

            var succeed = AssemblyUtility.TryGetFieldValueFromAssembly("Snapdragon.Spaces.Runtime", "Qualcomm.Snapdragon.Spaces.BaseRuntimeFeature", "LaunchControllerOnHost", feature, out var field);
            if (!succeed)
            {
                return false;
            }

            if (field.GetType() != typeof(bool))
            {
                return false;
            }

            result = (bool)field;
            return true;
        }

        public static bool TryGetExportHeadless(out bool result)
        {
            result = false;

            var settings = Resources.FindObjectsOfTypeAll<OpenXRSettings>().FirstOrDefault(s => s.name == "Android");
            if (settings == null)
            {
                return false;
            }

            var feature = settings.GetFeatures().FirstOrDefault(f => f.name == "BaseRuntimeFeature Android");
            if (feature == null)
            {
                return false;
            }

            var succeed = AssemblyUtility.TryGetFieldValueFromAssembly("Snapdragon.Spaces.Runtime", "Qualcomm.Snapdragon.Spaces.BaseRuntimeFeature", "ExportHeadless", feature, out var field);
            if (!succeed)
            {
                return false;
            }

            if (result.GetType() != typeof(bool))
            {
                return false;
            }

            result = (bool)field;
            return true;
        }

        public static bool TryGetIgnoreOpenXRVersion(out bool result)
        {
            result = false;

#if !USING_SPACES_1_0_2_OR_NEWER
            return false;
#endif

#pragma warning disable 162
            var settings = Resources.FindObjectsOfTypeAll<OpenXRSettings>().FirstOrDefault(s => s.name == "Android");
            if (settings == null)
            {
                return false;
            }

            var feature = settings.GetFeatures().FirstOrDefault(f => f.name == "FusionFeature Android");
            if (feature == null)
            {
                return false;
            }

            var succeed = AssemblyUtility.TryGetFieldValueFromAssembly("Snapdragon.Spaces.Runtime", "Qualcomm.Snapdragon.Spaces.FusionFeature", "IgnoreOpenXRVersion", feature, out var field);
            if (!succeed)
            {
                return false;
            }

            if (field.GetType() != typeof(bool))
            {
                return false;
            }

            result = (bool)field;
            return true;
#pragma warning restore 162
        }

        public static bool TryGetStartConnected(out bool result)
        {
            result = false;

            var target = Utility.FindFirstComponentByName("SpacesXRSimulator");
            if (target == null)
            {
                return false;
            }

            var succeed = AssemblyUtility.TryGetFieldValueFromAssembly("Snapdragon.Spaces.Runtime", "Qualcomm.Snapdragon.Spaces.SpacesXRSimulator", "StartConnected", target, out var field);
            if (!succeed)
            {
                return false;
            }

            if (field.GetType() != typeof(bool))
            {
                return false;
            }

            result = (bool)field;
            return true;
        }

        public static bool TrySetStartConnected(bool value, out MonoBehaviour target)
        {
            target = Utility.FindFirstComponentByName("SpacesXRSimulator");
            if (target == null)
            {
                return false;
            }

            var succeed = AssemblyUtility.TrySetFieldValueFromAssembly("Snapdragon.Spaces.Runtime", "Qualcomm.Snapdragon.Spaces.SpacesXRSimulator", "StartConnected", target, value);
            return succeed;
        }

        public static void ConfigureSpacesLifecycleEvents(MonoBehaviour spacesLifecycleEvents, bool isRenderCameraPresent)
        {
            Undo.RecordObject(spacesLifecycleEvents, "Register events to SpacesLifecycleEvents");

            AddSetActiveEvent("OnOpenXRAvailable", spacesLifecycleEvents, Constants.DrfNotifyConnectionPath, false);
            AddSetActiveEvent("OnOpenXRUnavailable", spacesLifecycleEvents, Constants.DrfNotifyConnectionPath, true);
            AddSetActiveEvent("OnOpenXRUnavailable", spacesLifecycleEvents, Constants.DrfControllerPointerPath, false);
            AddSetActiveEvent("OnOpenXRStarted", spacesLifecycleEvents, Constants.DrfControllerPointerPath, true);
            AddSetActiveEvent("OnOpenXRStopped", spacesLifecycleEvents, Constants.DrfControllerPointerPath, false);
            AddSetActiveEvent("OnHostViewEnabled", spacesLifecycleEvents,
                Constants.DrfCanvasControllerCompanionPath, true);
            AddSetActiveEvent("OnHostViewDisabled", spacesLifecycleEvents,
                Constants.DrfCanvasControllerCompanionPath, false);
            if (isRenderCameraPresent)
            {
                AddSetActiveEvent("OnHostViewDisabled", spacesLifecycleEvents, Constants.DrfRenderCameraPath, false);
            }
        }

        private static void AddSetActiveEvent(string fieldName, MonoBehaviour component, string targetPath, bool isActive)
        {
            GameObject target = null;

            if (targetPath == Constants.DrfRenderCameraPath)
            {
                // Find camera with target texture
                var cameras = Object.FindObjectsByType<Camera>(FindObjectsInactive.Include, FindObjectsSortMode.None);
                foreach (var c in cameras)
                {
                    if (c.targetTexture != null)
                    {
                        target = c.gameObject;
                        break;
                    }
                }
            }
            else
            {
                target = Utility.FindGameObjectByHierarchyPath(targetPath);
            }

            if (target == null)
            {
                Debug.LogError($"Target not found: {targetPath}");
                return;
            }

            var succeed = AssemblyUtility.TryGetFieldValueFromAssembly("Snapdragon.Spaces.Runtime", "Qualcomm.Snapdragon.Spaces.SpacesLifecycleEvents", fieldName, component, out var field);
            if (!succeed)
            {
                Debug.LogError($"Field not found: {fieldName}");
                return;
            }

            if (field is not UnityEvent unityEvent)
            {
                Debug.LogError($"Invalid type: {field.GetType().Name}");
                return;
            }

            var cnt = unityEvent.GetPersistentEventCount();
            var index = -1;
            for (var i = 0; i < cnt; i++)
            {
                var targetObject = unityEvent.GetPersistentTarget(i);
                if (targetObject == target)
                {
                    index = i;
                    break;
                }
            }

            var methodDelegate = (UnityAction<bool>)Delegate.CreateDelegate(typeof(UnityAction<bool>), target, AssemblyUtility.GetGameObjectSetActiveMethod());
            if (index == -1)
            {
                UnityEditor.Events.UnityEventTools.AddBoolPersistentListener(unityEvent, methodDelegate, isActive);
            }
            else
            {
                UnityEditor.Events.UnityEventTools.RegisterBoolPersistentListener(unityEvent, index, methodDelegate, isActive);
            }
            EditorUtility.SetDirty(component);
        }

        public static Component AddComponentToGameObject(GameObject target, string componentName, bool canUndo = false)
        {
            var fullPathClassName = $"Qualcomm.Snapdragon.Spaces.{componentName}";
            return AssemblyUtility.AddComponentToGameObject(target, "Snapdragon.Spaces.Runtime", fullPathClassName, canUndo);
        }

        public static bool RegisterCameraPermissionEventOnXRAvailable(MonoBehaviour spacesPermissionHelper)
        {
            var targetAssembly = AssemblyUtility.FindAssemblyByName("Snapdragon.Spaces.Runtime");
            var spacesPermissionHelperType = targetAssembly.GetType("Qualcomm.Snapdragon.Spaces.SpacesPermissionHelper");

            var result = AssemblyUtility.RegisterVoidUnityEvent(spacesPermissionHelper, spacesPermissionHelperType, "OnApplicationCameraPermissionsNotGranted", spacesPermissionHelper, spacesPermissionHelperType, "RequestApplicationCameraPermissions");

            return result;
        }

        public static bool TrySwitchAutoStartXRDisplayConnected(bool enable)
        {
            var loader = Utility.FindFirstComponentByName("DynamicOpenXRLoader");
            if (loader == null)
            {
                return false;
            }

            var targetAssembly = AssemblyUtility.FindAssemblyByName("Snapdragon.Spaces.Runtime");
            var targetType = targetAssembly.GetType("Qualcomm.Snapdragon.Spaces.DynamicOpenXRLoader");
            var fieldInfo = targetType.GetField("AutoStartXROnDisplayConnected", System.Reflection.BindingFlags.Public | System.Reflection.BindingFlags.Instance);
            if (fieldInfo != null)
            {
                fieldInfo.SetValue(loader, enable);
                EditorUtility.SetDirty(loader);
                return true;
            }
            else
            {
                return false;
            }
        }
    }

}