﻿using System;
using UnityEditor;
using UnityEngine;
using UnityEngine.UIElements;
using NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Runtime;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public class SimulatorPanel: BasePanel
    {
        public SimulatorPanel(VisualTreeAsset visualTreeAsset, ConfigurationWindow window) : base(visualTreeAsset, window)
        {
            AddOpenLinkButtonClickListener("OpenSimulatorManualButton","setup/mirza-sdk/categories/simulator");

            AddOpenLinkButtonClickListener("OpenSimulatorDocumentButton", "setup/scene/play-in-editor-setup-guide");

            AddButtonClickListener("AddSimulatorPrefabButton", () =>
            {
                GameObject gameObject = null;
                var c = Utility.FindFirstComponentByName("SpacesXRSimulator");
                if (c == null)
                {
                    bool result = false;
                    GameObject prefab = null;
                    if (AssemblyUtility.TryFindAssetPath(Constants.SpacesBasePath, Constants.SpacesXRSimulatorSearchPattern, out var path))
                    {
                        prefab = AssetDatabase.LoadAssetAtPath<GameObject>(path);
                        if (prefab != null)
                        {
                            result = true;
                        }
                    }
                    if (!result)
                    {
                        EditorUtility.DisplayDialog("Error", $"Prefab not found. Import spaces core samples.", "OK");
                        return;
                    }

                    var obj = PrefabUtility.InstantiatePrefab(prefab);
                    Undo.RegisterCreatedObjectUndo(obj, "Add Spaces XR Simulator");

                    gameObject = obj as GameObject;
                    if (gameObject == null)
                    {
                        EditorUtility.DisplayDialog("Error", "Failed to instantiate Spaces XR Simulator prefab.", "OK");
                        return;
                    }
                    gameObject.transform.localPosition = Vector3.zero;
                    Window.Validate();
                    EditorUtility.DisplayDialog("MiRZA Configuration Tool", "Spaces XR Simulator added to the scene.",
                        "OK");
                }
                else
                {
                    if (!SpacesAssemblyUtility.TrySetStartConnected(true, out c))
                    {
                        EditorUtility.DisplayDialog("Error", "Failed to set Start Connected to Spaces XR Simulator.", "OK");
                        return;
                    }
                    if (c != null)
                    {
                        gameObject = c.gameObject;
                        EditorUtility.SetDirty(c);
                    }
                    Window.Validate();
                    EditorUtility.DisplayDialog("MiRZA Configuration Tool", "Spaces XR Simulator already exists in the scene.\nStart Connected set to true.", "OK");
                }
                Selection.activeGameObject = gameObject;
            });

            AddOpenLinkButtonClickListener("OpenPointerSimulatorDocumentButton",
                "features/dual-render-fusion/setup/setup-guide/#companion-controller%E3%83%9D%E3%82%A4%E3%83%B3%E3%82%BF%E3%83%BC%E3%81%AE%E3%82%B7%E3%83%9F%E3%83%A5%E3%83%AC%E3%83%BC%E3%83%88");

            AddButtonClickListener("AddPointerSimulatorPrefabButton", () =>
            {
                var c = Utility.FindFirstComponentByName("PointerSimulator");
                if (c != null)
                {
                    EditorUtility.DisplayDialog("Error", "Pointer Simulator already exists in the scene.", "OK");
                    Selection.activeGameObject = c.gameObject;
                    return;
                }

                var prefabPath = Constants.PointerSimulatorPrefabPath;
                var prefab = AssetDatabase.LoadAssetAtPath<GameObject>(prefabPath);
                if (prefab == null)
                {
                    EditorUtility.DisplayDialog("Error", $"Prefab not found. Reimport MiRZA Configuration Tool.", "OK");
                    return;
                }

                var obj = PrefabUtility.InstantiatePrefab(prefab);
                Undo.RegisterCreatedObjectUndo(obj, "Add Pointer Simulator");
                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool", "Pointer Simulator added to the scene.", "OK");
                Selection.activeGameObject = obj as GameObject;
            });

            AddButtonClickListener("AssignPointerGameObjectButton", () =>
            {
                var obj = Utility.FindFirstComponentByName("PointerSimulator");
                var c = obj as PointerSimulator;
                if (c == null)
                {
                    EditorUtility.DisplayDialog("Error", "Pointer Simulator not found in the scene.", "OK");
                    return;
                }

                if (c.Pointer != null)
                {
                    EditorUtility.DisplayDialog("Error", "Pointer GameObject already assigned.", "OK");
                    Selection.activeGameObject = c.gameObject;
                    return;
                }

                Undo.RecordObject(c, "Assign Pointer GameObject");
                GameObject pointer =
                    Utility.FindGameObjectByHierarchyPathEndsWith(Constants.DrfControllerRelativePositionPath);
                if (pointer == null)
                {
                    EditorUtility.DisplayDialog("Error", "Dual Render Fusion Controller not found in the scene.", "OK");
                    return;
                }

                c.SetPointer(pointer);

                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool",
                    "Pointer GameObject assigned to Pointer Simulator.", "OK");
                Selection.activeGameObject = c.gameObject;
            });

            AddButtonClickListener("OpenHandTrackingSimulationSettingsButton", () =>
            {
                var info = Report.GetRequiredPackageInfo(Validator.InteractionsPackageName);
                if (info is not { Exists: true })
                {
                    EditorUtility.DisplayDialog("Error", "QCHT Unity Interactions package not found.", "OK");
                    return;
                }
                SettingsService.OpenProjectSettings("Project/XR Plug-in Management/Hand Tracking Simulation");
            });

            AddButtonClickListener("SetupSimulatorButton", () =>
            {
                var allWindows = Resources.FindObjectsOfTypeAll<EditorWindow>();
                EditorWindow projectSettingsWindow = null;
                foreach (var window in allWindows)
                {
                    if (window.titleContent != null && window.titleContent.text == "Project Settings")
                    {
                        projectSettingsWindow = window;
                        break;
                    }
                }

                if (projectSettingsWindow != null)
                {
                    if (!EditorUtility.DisplayDialog("MiRZA Configuration Tool", "The Project Settings window will be closed for setup.", "OK", "Cancel"))
                    {
                        return;
                    }
                    else
                    {
                        projectSettingsWindow.Close();
                    }
                }

                bool dualRenderFusionMode = Window.IsDualRenderFusionMode;
                try
                {
                    Utility.SetInitializeXrOnStartup(BuildTargetGroup.Android, !dualRenderFusionMode);
                    Utility.SetInitializeXrOnStartup(BuildTargetGroup.Standalone, !dualRenderFusionMode);
                    Utility.SetOpenXrPluginEnabled(BuildTargetGroup.Standalone, true);
                    Utility.SetXrSimulationPluginEnabled(BuildTargetGroup.Standalone, true);
                    Utility.SetBaseRuntimeFeatureEnabled(BuildTargetGroup.Standalone, true);
                }
                catch (Exception e)
                {
                    Debug.LogError("Failed to setup the simulator: " + e.Message);
                    if (EditorUtility.DisplayDialog("Error", "Failed to set up the simulator.\nPlease open Spaces Configuration Tool and configure before setup.", "Open", "Cancel"))
                    {
                        var info = Report.GetRequiredPackageInfo(Validator.SpacesPackageName);
                        if (info is not { Exists: true })
                        {
                            EditorUtility.DisplayDialog("Error", "Snapdragon Spaces package not found.", "OK");
                            return;
                        }
                        SpacesAssemblyUtility.ShowSpacesSetupWindow();
                    }
                    return;
                }

                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool", "Setup the Simulator.", "OK");
            });

            AddOpenLinkButtonClickListener("OpenHandTrackingSimulatorDocumentButton",
                "tips/editor-simulation/#%E3%83%8F%E3%83%B3%E3%83%89%E3%83%88%E3%83%A9%E3%83%83%E3%82%AD%E3%83%B3%E3%82%B0%E3%81%AE%E3%82%B7%E3%83%BC%E3%83%B3%E3%81%AE%E5%A0%B4%E5%90%88");

            AddButtonClickListener("CreateMirzaSimulationGameViewButton",
                () => { CameraSimulator.ConfigureSimulator(); });

            AddButtonClickListener("AddMirzaDeviceEventSimulatorButton", () =>
            {
#if USING_MIRZA_SDK
                var c = Utility.FindFirstComponentByName("MirzaDeviceEventSimulator");
                if (c != null)
                {
                    EditorUtility.DisplayDialog("Error", "Mirza Device Event Simulator already exists in the scene.", "OK");
                    Selection.activeGameObject = c.gameObject;
                    return;
                }
                var gameObject = new GameObject("MirzaDeviceEventSimulator");
                Undo.RegisterCreatedObjectUndo(gameObject, "Add Mirza Device Event Simulator");
                gameObject.AddComponent<MirzaDeviceEventSimulator>();

                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool", "Mirza Device Event Simulator added to the scene.", "OK");
                Selection.activeGameObject = gameObject;
#else
                EditorUtility.DisplayDialog("Error", "MiRZA Library must be imported before using the Mirza Device Event Simulator.", "OK");
#endif
            });

            AddOpenLinkButtonClickListener("OpenMiRZALibraryEventButton", "setup/mirza-sdk/mirza-library/mirzalibrary/interfaces/package-summary/");
        }

        public override void RefreshPanel()
        {
            UpdateValueLabelByName("InitializeXROnStartupPCLabel", Report.SimulatorParams.InitializeXROnStartupPC);
            UpdateValueLabelByName("InitializeXROnStartupAndroidLabel",
                Report.SimulatorParams.InitializeXROnStartupAndroid, true);
            UpdateValueLabelByName("SpacesXRSimulatorLabel", Report.SimulatorParams.SpacesXRSimulator);
            UpdateValueLabelByName("StartConnectedLabel", Report.SimulatorParams.StartConnected);
            UpdateValueLabelByName("OpenXRLabel", Report.SimulatorParams.OpenXRPluginEnabled);
            UpdateValueLabelByName("XRSimulationLabel", Report.SimulatorParams.XRSimulationPluguinEnabled);
            UpdateValueLabelByName("BaseRuntimeLabel", Report.SimulatorParams.BaseRuntimeFeatureEnabled);
            UpdateValueLabelByName("PointerSimulatorLabel", Report.SimulatorParams.PointerSimulator);
            UpdateValueLabelByName("XRDeviceSimulatorLabel", Report.SimulatorParams.XRDeviceSimulatorEnabled);
            UpdateValueLabelByName("HandTrackingSimulationLabel", Report.SimulatorParams.HandTrackingSimulationEnabled);
            UpdateValueLabelByName("MirzaDeviceEventSimulatorLabel", Report.SimulatorParams.MirzaDeviceEventSimulator);

            var pointerGameObjectParam = Report.SimulatorParams.PointerGameObject;
            if (pointerGameObjectParam.Exists)
            {
                PanelParent.Q<VisualElement>("PointerGameObjectLabel").tooltip = pointerGameObjectParam.Actual;
                var pointerGameObjectName = pointerGameObjectParam.Actual;
                if (pointerGameObjectName.Contains("/"))
                {
                    pointerGameObjectName = pointerGameObjectName.Substring(pointerGameObjectName.LastIndexOf("/") + 1);
                }

                pointerGameObjectParam.Actual = $".../{pointerGameObjectName}";
            }

            UpdateValueLabelByName("PointerGameObjectLabel", pointerGameObjectParam);
        }
    }
}