﻿using System.IO;
using System.Text;
using Unity.XR.CoreUtils;
using UnityEditor;
using UnityEngine;
using UnityEngine.UIElements;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public class SceneSetupPanel : BasePanel
    {
        public SceneSetupPanel(VisualTreeAsset visualTreeAsset, ConfigurationWindow window) : base(visualTreeAsset, window)
        {
            AddOpenLinkButtonClickListener("OpenSceneSetupManualButton", "setup/mirza-sdk/categories/scene-setup");

            AddButtonClickListener("SetupXRCoreModulesButton", () =>
            {
                // Add AR Session
                var arSession = Utility.FindFirstComponentByName("ARSession");
                if (arSession == null)
                {
                    EditorApplication.ExecuteMenuItem("GameObject/XR/AR Session");
                    arSession = Utility.FindFirstComponentByName("ARSession");
                    Undo.RegisterCreatedObjectUndo(arSession.gameObject, "Add AR Session");
                    arSession.gameObject.transform.localPosition = Vector3.zero;
                }

                //Add XR Origin
                var xrOriginObject = Utility.FindFirstComponentByName("XROrigin");
                if (xrOriginObject == null)
                {
#if USING_XRI_3_0_0_PRE_1_OR_NEWER
                    EditorApplication.ExecuteMenuItem("GameObject/XR/XR Origin (Mobile AR)");
#else
                    EditorApplication.ExecuteMenuItem("GameObject/XR/XR Origin (AR)");
#endif
                    xrOriginObject = Utility.FindFirstComponentByName("XROrigin");
                    Undo.RegisterCreatedObjectUndo(xrOriginObject.gameObject, "Add XR Origin");
                    xrOriginObject.gameObject.transform.localPosition = Vector3.zero;
                }
                var xrOrigin = xrOriginObject as XROrigin;
                if (xrOrigin != null)
                {
                    xrOrigin.RequestedTrackingOriginMode = XROrigin.TrackingOriginMode.Device;
                    xrOrigin.CameraYOffset = 0f;
                    EditorUtility.SetDirty(xrOrigin);
                }

                var xrOriginCamera = xrOrigin.Camera;
                if (xrOriginCamera != null)
                {
                    xrOriginCamera.targetDisplay = 1;
                    xrOriginCamera.clearFlags = CameraClearFlags.SolidColor;
                    xrOriginCamera.backgroundColor = Color.black;
                    EditorUtility.SetDirty(xrOriginCamera);
                }

                var cameraOffset = xrOriginObject.transform.Find("Camera Offset");
                if (cameraOffset != null)
                {
                    cameraOffset.localPosition = Vector3.zero;
                }

                var xrInteractionManager = Utility.FindComponentsByName("XRInteractionManager");
                foreach (var manager in xrInteractionManager)
                {
                    manager.gameObject.transform.localPosition = Vector3.zero;
                }

                // Disable AR Camera Manager
                var arCameraManager = Utility.FindComponentsByName("ARCameraManager");
                foreach (var manager in arCameraManager)
                {
                    Undo.RecordObject(manager, "Disable AR Camera Manager");
                    manager.enabled = false;
                }

                // Disable AR Camera Background
                var arCameraBackground = Utility.FindComponentsByName("ARCameraBackground");
                foreach (var background in arCameraBackground)
                {
                    Undo.RecordObject(background, "Disable AR Camera Background");
                    background.enabled = false;
                }

                // Delete Default Controllers
                var leftControllerTarget =
                    Utility.FindGameObjectByHierarchyPathEndsWith(Constants.DefaultLeftControllerGameObjectPath);
                var rightControllerTarget =
                    Utility.FindGameObjectByHierarchyPathEndsWith(Constants.DefaultRightControllerGameObjectPath);

                if (leftControllerTarget != null)
                {
                    Undo.DestroyObjectImmediate(leftControllerTarget);
                }
                if (rightControllerTarget != null)
                {
                    Undo.DestroyObjectImmediate(rightControllerTarget);
                }

                // Delete Main Camera GameObject that does not have XR Origin as its parent
                var mainCameras = Utility.FindGameObjectsByTag("MainCamera");
                foreach (var mainCamera in mainCameras)
                {
                    var isARFoundationCamera = mainCamera.GetComponentInParent<XROrigin>() != null;
                    if (!isARFoundationCamera)
                    {
                        Undo.DestroyObjectImmediate(mainCamera);
                    }
                }
                Debug.Log("Deleted Main Camera GameObject that does not have XR Origin as its parent.");

                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool",
                    "XR Core Modules setup completed.", "OK");
                EditorUtility.FocusProjectWindow();
            });

            AddButtonClickListener("AddSpacesModulesButton", () =>
            {
                // Check importing Snapdragon Spaces
                var param = Report.GetRequiredPackageInfo(Validator.SpacesPackageName);
                if (!param.Exists)
                {
                    EditorUtility.DisplayDialog("Error",
                        "Snapdragon Spaces SDK not found.\nImport Snapdragon Spaces SDK first.", "OK");
                    return;
                }

                // Add Dynamic Open XR Loader
                var dynamicOpenXRLoader = Utility.FindFirstComponentByName("DynamicOpenXRLoader");
                if (dynamicOpenXRLoader == null)
                {
                    EditorApplication.ExecuteMenuItem("GameObject/XR/Snapdragon Spaces/Dual Render Fusion/Dynamic OpenXR Loader");
                    dynamicOpenXRLoader = Utility.FindFirstComponentByName("DynamicOpenXRLoader");
                    Undo.RegisterCreatedObjectUndo(dynamicOpenXRLoader.gameObject, "Add Dynamic OpenXR Loader");
                }

                // Add Spaces Host View
                var spacesHostView = Utility.FindFirstComponentByName("SpacesHostView");
                if (spacesHostView == null)
                {
                    EditorApplication.ExecuteMenuItem("GameObject/XR/Snapdragon Spaces/Dual Render Fusion/Host View");
                    spacesHostView = Utility.FindFirstComponentByName("SpacesHostView");
                    Undo.RegisterCreatedObjectUndo(spacesHostView.gameObject, "Add Spaces Host View");
                }

                // Add Spaces Lifecycle Events
                var spacesLifecycleEvents = Utility.FindFirstComponentByName("SpacesLifecycleEvents");
                if (spacesLifecycleEvents == null)
                {
                    EditorApplication.ExecuteMenuItem("GameObject/XR/Snapdragon Spaces/Dual Render Fusion/Spaces Lifecycle Events");
                    spacesLifecycleEvents = Utility.FindFirstComponentByName("SpacesLifecycleEvents");
                    Undo.RegisterCreatedObjectUndo(spacesLifecycleEvents.gameObject, "Add Spaces Lifecycle Events");
                }

                // Add Spaces XR Simulator
                var spacesXRSimulator = Utility.FindFirstComponentByName("SpacesXRSimulator");
                if (spacesXRSimulator == null)
                {
                    EditorApplication.ExecuteMenuItem("GameObject/XR/Snapdragon Spaces/Spaces XR Simulator");
                    spacesXRSimulator = Utility.FindFirstComponentByName("SpacesXRSimulator");
                    Undo.RegisterCreatedObjectUndo(spacesXRSimulator.gameObject, "Add Spaces XR Simulator");
                    MonoBehaviour component = null;
                    if (!SpacesAssemblyUtility.TrySetStartConnected(true, out component))
                    {
                        EditorUtility.DisplayDialog("Error", "Failed to set Start Connected to Spaces XR Simulator.", "OK");
                    }
                    if (component != null)
                    {
                        EditorUtility.SetDirty(component);
                    }
                }
                else
                {
                    if (!SpacesAssemblyUtility.TrySetStartConnected(true, out spacesXRSimulator))
                    {
                        EditorUtility.DisplayDialog("Error", "Failed to set Start Connected to Spaces XR Simulator.", "OK");
                        return;
                    }
                    if (spacesXRSimulator != null)
                    {
                        EditorUtility.SetDirty(spacesXRSimulator);
                    }
                }

                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool",
                    "Snapdragon Spaces Modules added to the scene.", "OK");
                EditorUtility.FocusProjectWindow();
            });

            AddButtonClickListener("AddDualRenderFusionControllerButton", () =>
            {
                if (!Utility.TryAddDualRenderFusionControllerPrefabToScene()) return;
                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool",
                    "Dual Render Fusion Controller added to the scene.", "OK");
            });

            AddButtonClickListener("AddRenderCameraButton", () =>
            {
                if (!Utility.TryAddRenderCameraPrefabToScene()) return;
                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool", "Render Camera added to the scene.", "OK");
            });

            AddButtonClickListener("AddEventsToSpacesLifecycleEventsButton", () =>
            {
                Window.Validate();
                var sle = Utility.FindFirstComponentByName("SpacesLifecycleEvents");
                if (sle == null)
                {
                    EditorUtility.DisplayDialog("Error", "SpacesLifecycleEvents not found in the scene.", "OK");
                    return;
                }

                var controller = Utility.FindGameObjectByName(Constants.ControllerGameObjectName);
                if (controller == null)
                {
                    EditorUtility.DisplayDialog("Error",
                        $"{Constants.ControllerGameObjectName} prefab not found in the scene.\nTo add {Constants.ControllerGameObjectName} to the scene,\nplease press the 'Add Dual Render Fusion Controller' button first.", "OK");
                    return;
                }

                bool isRenderCameraPresent = Utility.FindGameObjectByName(Constants.RenderCameraGameObjectName) != null;
                if (!isRenderCameraPresent)
                {
                    Debug.LogWarning($"{Constants.RenderCameraGameObjectName} prefab not found in the scene. If you want to use {Constants.RenderCameraGameObjectName}, please press the 'Add Render Camera' button and then try again.");
                }

                Selection.activeGameObject = sle.gameObject;
                if (EditorUtility.DisplayDialog("MiRZA Configuration Tool",
                        "Register events to SpacesLifecycleEvents", "OK", "Cancel"))
                {
                    SpacesAssemblyUtility.ConfigureSpacesLifecycleEvents(sle, isRenderCameraPresent);
                }
            });

            AddButtonClickListener("SetupInteractionManagersButton", () =>
            {
                GameObject targetObject = null;

                var xrInteractionManagers = Utility.FindComponentsByName("XRInteractionManager");
                if (xrInteractionManagers != null && xrInteractionManagers.Count > 0)
                {
                    targetObject = xrInteractionManagers[0].gameObject;
                }
                else
                {
                    targetObject = new GameObject("XR Interaction Manager");
                    Undo.RegisterCreatedObjectUndo(targetObject, "Add XR Interaction Manager");

                    var fullPathClassName = $"UnityEngine.XR.Interaction.Toolkit.XRInteractionManager";
                    var component = AssemblyUtility.AddComponentToGameObject(targetObject, "Unity.XR.Interaction.Toolkit", fullPathClassName);
                    if (component == null)
                    {
                        EditorUtility.DisplayDialog("Error", "Failed to add XR Interaction Manager component.\nReimport the XR Interaction Toolkit.", "OK");
                        return;
                    }
                }

                if (XRInteractionToolkitAssemblyUtility.TryGetConfiguredInputActionManager(out var inputActionManager))
                {
                    targetObject = inputActionManager.gameObject;
                }
                else
                {
                    Component component = null;
                    var inputActionManagers = Utility.FindComponentsByName("InputActionManager");
                    if (inputActionManagers != null && inputActionManagers.Count > 0)
                    {
                        targetObject = inputActionManagers[0].gameObject;
                        component = Utility.FindAttachedComponentByName(targetObject, "InputActionManager");
                    }
                    else
                    {
                        var fullPathClassName = $"UnityEngine.XR.Interaction.Toolkit.Inputs.InputActionManager";
                        component = AssemblyUtility.AddComponentToGameObject(targetObject, "Unity.XR.Interaction.Toolkit", fullPathClassName, true);
                        if (component == null)
                        {
                            EditorUtility.DisplayDialog("Error", "Failed to add Input Action Manager component.\nReimport the XR Interaction Toolkit.", "OK");
                            return;
                        }
                    }

                    Undo.RecordObject(component, "Set Default QCHT Input Actions");
                    if (!XRInteractionToolkitAssemblyUtility.TrySetDefaultQCHTInputActions(component))
                    {
                        EditorUtility.DisplayDialog("Error", "Failed to set Default QCHT Input Actions to Input Action Manager.\nImport the Core Assets of QCHT Unity Interactions.", "OK");
                        return;
                    }
                    EditorUtility.SetDirty(component);
                }

                EditorUtility.DisplayDialog("MiRZA Configuration Tool", "Interaction Managers setup completed.", "OK");
                Window.Validate();
                Selection.activeGameObject = targetObject;
            });

            AddButtonClickListener("AddQCHTXROriginButton", () =>
            {
                // Add QCHT XR origin
                var target = Utility.FindGameObjectByName(Constants.QCHTXROriginGameObjectName);
                if (target != null)
                {
                    EditorUtility.DisplayDialog("Error",
                        $"{Constants.QCHTXROriginGameObjectName} already exists in the scene.", "OK");
                    Selection.activeGameObject = target;
                    return;
                }

                GameObject prefab = null;
                string[] directories = Directory.GetDirectories(Constants.QCHTBasePath, "*",
                    SearchOption.AllDirectories);
                foreach (string dir in directories)
                {
                    string potentialPath = Path.Combine(dir, Constants.QCHTXROriginPrefabSearchPattern);
                    if (File.Exists(potentialPath))
                    {
                        prefab = AssetDatabase.LoadAssetAtPath<GameObject>(potentialPath);
                        if (prefab != null)
                        {
                            Debug.Log("Prefab loaded successfully from: " + potentialPath);
                            break;
                        }
                    }
                }
                if (prefab == null)
                {
                    EditorUtility.DisplayDialog("Error", $"Prefab not found.\nImport the Core Assets of QCHT Unity Interactions.", "OK");
                    return;
                }
                var obj = PrefabUtility.InstantiatePrefab(prefab);
                Undo.RegisterCreatedObjectUndo(obj, "Add QCHT XR Origin");

                // Setting XR Origin
                target = obj as GameObject;
                var xrOrigin = target.GetComponent<XROrigin>();
                Undo.RecordObject(xrOrigin, "Update xrOrigin Settings");
                xrOrigin.RequestedTrackingOriginMode = XROrigin.TrackingOriginMode.Device;
                xrOrigin.CameraYOffset = 0f;
                EditorUtility.SetDirty(xrOrigin);

                var mainCamera = xrOrigin.Camera;
                Undo.RecordObject(mainCamera, "Update Main Camera Settings");
                mainCamera.clearFlags = CameraClearFlags.SolidColor;
                mainCamera.backgroundColor = Color.black;
                mainCamera.targetDisplay = 1;
                EditorUtility.SetDirty(mainCamera);

                var cameraOffset = xrOrigin.transform.Find("Camera Offset");
                if (cameraOffset != null)
                {
                    cameraOffset.localPosition = Vector3.zero;
                }

                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool",
                    "QCHT XR Origin added to the scene.", "OK");
                Selection.activeGameObject = target;
            });

            AddButtonClickListener("DeleteXROriginExceptQCHTButton", () =>
            {
                var xrOrigins = Utility.FindComponentsByName("XROrigin");
                foreach (var xrOrigin in xrOrigins)
                {
                    if (!xrOrigin.gameObject.name.Contains(Constants.QCHTXROriginGameObjectName))
                    {
                        Undo.DestroyObjectImmediate(xrOrigin.gameObject);
                    }
                }

                Window.Validate();
                EditorUtility.DisplayDialog("MiRZA Configuration Tool",
                    "XR Origin objects except for QCHT XR Origin deleted.", "OK");
            });

            AddOpenLinkButtonClickListener("OpenSceneSetupGuideButton", "setup/scene/scene-setup");

            AddOpenLinkButtonClickListener("OpenDualRenderFusionGuideButton", "features/dual-render-fusion/setup/components");

            AddOpenLinkButtonClickListener("QCHTControlBoxGuideButton", "features/hand-tracking/interaction-features/control-box/");

            AddButtonClickListener("SetupCanvasesForXRButton", () =>
            {
                var xrOriginObject = Utility.FindFirstComponentByName("XROrigin");
                if (xrOriginObject == null)
                {
                    EditorUtility.DisplayDialog("Error",
                        "XR Origin not found in the scene.\nAdd the XR Origin to the scene first.", "OK");
                    return;
                }

                if (!EditorUtility.DisplayDialog("MiRZA Configuration Tool",
                        "This operation will set all canvases for XR.\nDo you want to continue?",
                        "OK", "Cancel"))
                {
                    return;
                }

                var xrCamera = (xrOriginObject as XROrigin).Camera;
                var controller = Utility.FindGameObjectByHierarchyPath(Constants.DrfCanvasControllerCompanionPath);
                var controllerCanvas = controller?.GetComponentInChildren<Canvas>(true);

                string undoName = "Set Canvases for XR";
                Undo.SetCurrentGroupName(undoName);

                // controller以外の全てのcanvasを持つGameObjectを取得
                var canvases = Utility.FindComponentsByType<Canvas>();

                foreach (var canvas in canvases)
                {
                    if (canvas.gameObject != controllerCanvas?.gameObject)
                    {
                        Undo.RecordObject(canvas, undoName);
                        canvas.renderMode = RenderMode.WorldSpace;
                        canvas.worldCamera = xrCamera;

                        XRInteractionToolkitAssemblyUtility.AddTrackedDeviceGraphicRaycaster(canvas);

                        EditorUtility.SetDirty(canvas.gameObject);
                        Selection.activeGameObject = canvas.gameObject;
                    }
                }
                EditorUtility.DisplayDialog("MiRZA Configuration Tool", "Set Canvases for XR completed.", "OK");

            });

            AddOpenLinkButtonClickListener("OpenCameraPermissionGuideButton", "tips/application-lifecycle/handling-app-permissions/fusion");

            AddButtonClickListener("SetupForRequestCameraPermissionOnStartButton", () =>
            {
                var sleList = Utility.FindComponentsByName("SpacesLifecycleEvents");
                if (sleList.Count == 0)
                {
                    EditorUtility.DisplayDialog("Error", "SpacesLifecycleEvents not found in the scene.", "OK");
                    return;
                }

                MonoBehaviour helper;
                var speList = Utility.FindComponentsByName("SpacesPermissionHelper");
                if (speList.Count == 0)
                {
                    // 存在しない場合はSpacesPermissionHelperのオブジェクトを新規作成
                    EditorUtility.DisplayDialog("MiRZA Configuration Tool", "SpacesPermissionHelper is added to the scene.", "OK");
                    GameObject gameObject = new GameObject("SpacesPermissionHelper");
                    helper = (MonoBehaviour)SpacesAssemblyUtility.AddComponentToGameObject(gameObject, "SpacesPermissionHelper");
                    Undo.RegisterCreatedObjectUndo(gameObject, "Add Space Permission Helper");
                }
                else
                {
                    helper = speList[0];
                    Undo.RecordObject(helper, "Register event to Spaces Permission Helper");
                }
                var result = SpacesAssemblyUtility.RegisterCameraPermissionEventOnXRAvailable(helper);

                if (result)
                {
                    EditorUtility.DisplayDialog("MiRZA Configuration Tool", "UnityEvent listener added successfully.",
                        "OK");
                    EditorUtility.SetDirty(helper);
                }
                Window.Validate();
                Selection.activeGameObject = helper.gameObject;
            });

            AddButtonClickListener("OpenTemplatesButton", () => { Window.MenuListView.selectedIndex = 3; });

            AddButtonClickListener("SetupForMRModeAutoSwitchButton", () =>
            {
                // packageの有無の確認
                var mirzaLibrary = Report.GetRequiredPackageInfo(Validator.MirzaLibraryPackageName);
                if (!mirzaLibrary.Exists)
                {
                    EditorUtility.DisplayDialog("Error",
                        "MiRZA Library not found.\nImport MiRZA Library.", "OK");
                    return;
                }
                var spacesSdk = Report.GetRequiredPackageInfo(Validator.SpacesPackageName);
                if (!spacesSdk.Exists)
                {
                    EditorUtility.DisplayDialog("Error",
                        "Snapdragon Spaces SDK not found.\nImport Snapdragon Spaces SDK.", "OK");
                    return;
                }

                if (!SpacesAssemblyUtility.TrySwitchAutoStartXRDisplayConnected(false))
                {
                    EditorUtility.DisplayDialog("Error",
                        "Failed to set DynamicOpenXRLoader.AutoStartXROnDisplayConnected to false.", "OK");
                    return;
                }


                var target = Utility.FindGameObjectByName(Constants.MRModeAutoSwitcherGameObjectName);
                if (target != null)
                {
                    EditorUtility.DisplayDialog("Error",
                        $"{Constants.MRModeAutoSwitcherGameObjectName} already exists in the scene.", "OK");
                    Selection.activeGameObject = target;
                    return;
                }

                StringBuilder sb = new StringBuilder();
                bool isOverwrite = false;

                sb.AppendLine("Files below will be overwritten:");
                sb.AppendLine();

                var copyHelper = new MRModeAutoSwitcherCopyHelper();


                var scriptSourcePath = copyHelper.SourceScriptPath;
                var scriptTargetPath = $"{AssetUtility.AssetPath}/{AssetUtility.ScriptsSubFolderName}/MrModeSwitcher.cs";

                if (File.Exists(scriptTargetPath))
                {
                    sb.AppendLine($"  - {scriptTargetPath}");
                    isOverwrite = true;
                }

                var prefabSourcePath = copyHelper.SourcePrefabPath;
                var prefabTargetPath = $"{AssetUtility.AssetPath}/{AssetUtility.PrefabSubFolderName}/MR Mode Auto Switcher.prefab";

                if (File.Exists(scriptTargetPath))
                {
                    sb.AppendLine($"  - {prefabTargetPath}");
                    isOverwrite = true;
                }

                sb.AppendLine();
                sb.AppendLine("This operation does not support Undo.");

                if (isOverwrite)
                {
                    if (EditorUtility.DisplayDialog("MiRZA Configuration Tool", sb.ToString(), "OK", "Cancel"))
                    {
                        AssetUtility.CopyAsset(scriptSourcePath, scriptTargetPath);
                        AssetUtility.CopyAsset(prefabSourcePath, prefabTargetPath);
                    }
                    else
                    {
                        return;
                    }
                }
                else
                {
                    EditorUtility.DisplayDialog("MiRZA Configuration Tool", "MR Mode Auto Switcher will be loaded.", "OK");
                    AssetUtility.CopyAsset(scriptSourcePath, scriptTargetPath);
                    AssetUtility.CopyAsset(prefabSourcePath, prefabTargetPath);
                }
                sb.Clear();
                AssetDatabase.Refresh();

                copyHelper.RebuildMRModeAutoSwitcherPrefab(prefabTargetPath, scriptTargetPath);

                Window.Validate();

            });

            AddButtonClickListener("OpenTemplatesButton0", () => { Window.MenuListView.selectedIndex = 3; });

            AddButtonClickListener("OpenSamplesButton0", () => { Window.MenuListView.selectedIndex = 4; });

            AddButtonClickListener("OpenQCHTSamplesButton", () =>
            {
                var info = Report.GetRequiredPackageInfo(Validator.InteractionsPackageName);
                if (info is not { Exists: true })
                {
                    EditorUtility.DisplayDialog("Error", "QCHT Unity Interactions package not found.", "OK");
                    return;
                }
                UnityEditor.PackageManager.UI.Window.Open(Validator.InteractionsPackageName);
                Utility.OpenSamplesTabInPackageManagerWindow();
            });
        }


        public override void RefreshPanel()
        {
            UpdateValueLabelByName("ARSessionLabel", Report.SceneParams.ARSession, true);

            UpdateValueLabelByName("XROriginLabel", Report.SceneParams.XROrigin, true);

            UpdateValueLabelExpectedNotFoundOrDisabled("ARCameraManagerLabel", Report.SceneParams.ARCameraManager);

            UpdateValueLabelExpectedNotFoundOrDisabled("ARCameraBackgroundLabel", Report.SceneParams.ARCameraBackground);

            UpdateValueLabelByName("DynamicOpenXRLoaderLabel", Report.SceneParams.DynamicOpenXRLoader, true);

            UpdateValueLabelByName("SpacesHostViewLabel", Report.SceneParams.SpacesHostView, true);

            UpdateValueLabelByName("SpacesLifecycleEventsLabel", Report.SceneParams.SpacesLifecycleEvents);

            UpdateValueLabelByName("SpacesXRSimulatorLabel", Report.SceneParams.SpacesXRSimulator);

            UpdateValueLabelByName("DualRenderFusionControllerLabel", Report.SceneParams.DualRenderFusionController);

            UpdateValueLabelByName("RenderCameraLabel", Report.SceneParams.RenderCamera);

            UpdateValueLabelByName("SpacesLifecycleEventConfiguration_SpecesLifecycleEventsLabel", Report.SceneParams.SpacesLifecycleEvents);

            UpdateValueLabel("QchtSdkVersionLabel", Report.GetRequiredPackageInfo(Validator.InteractionsPackageName));

            UpdateValueLabel("QchtSdkCoreAssetsLabel",
                Report.GetRequiredAssetInfo(Validator.InteractionsPackageName, "Core Assets"));

            UpdateValueLabelByName("HandTrackingFeatureLabel", Report.TemplatesParams.HandTrackingFeatureEnabled);

            UpdateValueLabelByName("XRInteractionManagerLabel", Report.SceneParams.XRInteractionManager);

            UpdateValueLabelByName("ConfiguredInputActionManagerLabel", Report.SceneParams.ConfiguredInputActionManager);

            UpdateValueLabelByName("QCHTXROriginLabel", Report.SceneParams.QCHTXROrigin);

            UpdateValueLabel("QCHTControlBoxSupport_QchtSdkVersionLabel", Report.GetRequiredPackageInfo(Validator.InteractionsPackageName));

            UpdateValueLabelByName("SetupForCameraPermission_SpecesLifecycleEventsLabel", Report.SceneParams.SpacesLifecycleEvents);

            UpdateValueLabelByName("SpecesPermissionHelperLabel", Report.SceneParams.SpacesPermissionHelper);

            UpdateValueLabelByName("MRModeAutoSwitcherLabel", Report.SceneParams.MRModeAutoSwitcher);

            var cameraList = PanelParent.Q<VisualElement>("CameraList");
            if (cameraList != null)
            {
                cameraList.Clear();
                foreach (var cameraInfo in Report.SceneParams.Cameras)
                {
                    var cameraListItem = Window.CameraListItemAsset.Instantiate();
                    cameraList.Add(cameraListItem);

                    var cameraNameLabel = cameraListItem.Q<VisualElement>("CameraNameLabel");
                    cameraNameLabel.Q<Label>("ValueLabel").Set(cameraInfo.Name, DefaultValueLabelColor);

                    var scenePathLabel = cameraListItem.Q<VisualElement>("ScenePathLabel");
                    scenePathLabel.Q<Label>("ValueLabel").Set(cameraInfo.ScenePath, DefaultValueLabelColor);

                    var tagLabel = cameraListItem.Q<VisualElement>("TagLabel");
                    tagLabel.Q<Label>("ValueLabel").Set(cameraInfo.Tag, DefaultValueLabelColor);

                    var layerLabel = cameraListItem.Q<VisualElement>("LayerLabel");
                    layerLabel.Q<Label>("ValueLabel").Set(cameraInfo.Layer, DefaultValueLabelColor);

                    var targetDisplayLabel = cameraListItem.Q<VisualElement>("TargetDisplayLabel");
                    targetDisplayLabel.Q<Label>("NameLabel").text = "Target Display" + (cameraInfo.IsXrOriginCamera ? " (Recommended Target : Display 2)" : "");
                    UpdateValueLabel(targetDisplayLabel.Q<Label>("ValueLabel"), cameraInfo.TargetDisplay);

                    var targetEyeLabel = cameraListItem.Q<VisualElement>("TargetEyeLabel");
                    var targetEyeValueLabelColor = DefaultValueLabelColor;
                    targetEyeLabel.Q<Label>("NameLabel").text = "Target Eye" + (cameraInfo.IsUrpPhoneCamera ? " (Recommended Target : None)" : "");
                    if (cameraInfo.IsUrpPhoneCamera && !cameraInfo.TargetEye.Equals(Constants.TargetEyeIsNoneInUrpProject))
                    {
                        targetEyeValueLabelColor = ErrorValueLabelColor;
                    }
                    targetEyeLabel.Q<Label>("ValueLabel").Set(cameraInfo.TargetEye, targetEyeValueLabelColor);

                    var activeLabel = cameraListItem.Q<VisualElement>("ActiveLabel");
                    activeLabel.Q<Label>("ValueLabel").Set(cameraInfo.IsActive.ToString(), DefaultValueLabelColor);

                    var activeSelfLabel = cameraListItem.Q<VisualElement>("ActiveSelfLabel");
                    activeSelfLabel.Q<Label>("ValueLabel").Set(cameraInfo.IsActiveSelf.ToString(), DefaultValueLabelColor);

                    cameraListItem.Q<Button>("SelectButton").clickable.clicked += () =>
                    {
                        Selection.activeGameObject = cameraInfo.TargetCamera?.gameObject;
                    };
                }
            }
        }
    }
}