﻿using System;
using System.IO;
using System.Linq;
using TMPro;
using UnityEditor;
using UnityEditor.Build;
using UnityEditor.Compilation;
using UnityEditor.PackageManager;
using UnityEditor.PackageManager.Requests;
using UnityEngine;
using UnityEngine.UIElements;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public class ProjectSetupPanel : BasePanel
    {
        private AddRequest _addMirzaLibraryRequest;
        private AddRequest _addOpenXRPluginRequest;

        private static readonly string OpenXRTitle = "OpenXR Plugin ";
        private static readonly string OpenXRRecommendedVersion = "(Recommended Version : 1.10.0)";
        private static readonly string OpenXRIgnoreVersionStatus = "(Ignore OpenXR Version : True)";

        public ProjectSetupPanel(VisualTreeAsset visualTreeAsset, ConfigurationWindow window) : base(visualTreeAsset, window)
        {
            AddOpenLinkButtonClickListener("OpenProjectSetupManualButton", "setup/mirza-sdk/categories/project-setup");

            AddButtonClickListener("OpenSpacesConfigButton", () =>
            {
                var info = Report.GetRequiredPackageInfo(Validator.SpacesPackageName);
                if (info is not { Exists: true })
                {
                    EditorUtility.DisplayDialog("Error", "Snapdragon Spaces package not found.", "OK");
                    return;
                }

                SpacesAssemblyUtility.ShowSpacesSetupWindow();
            });

            AddOpenLinkButtonClickListener(
                "OpenSetupGuideButton",
                "setup/setup-guide"
            );

            AddOpenLinkButtonClickListener(
                "OpenSpacesImportGuideButton",
                "setup/spaces-sdk/spaces-sdk-setup#sdk%E3%81%AE%E3%83%80%E3%82%A6%E3%83%B3%E3%83%AD%E3%83%BC%E3%83%89"
            );

            AddOpenLinkButtonClickListener(
                "OpenSpacesConfigGuideButton",
                "setup/spaces-sdk/configuration-tool/configuration-tool-drf"
            );

            AddButtonClickListener("ImportTMPEssentialButton", () =>
            {
                EditorApplication.ExecuteMenuItem("Window/TextMeshPro/Import TMP Essential Resources");
            });

            AddButtonClickListener("ImportControllerButton", () =>
            {
                var param = Report.GetRequiredPackageInfo(Validator.ConfigToolPackageName);
                var displayName = "Dual Render Fusion Controller";

                var spacesSdkParam = Report.GetRequiredPackageInfo(Validator.SpacesPackageName);
                if (spacesSdkParam is not { Exists: true })
                {
                    EditorUtility.DisplayDialog("Error", $"{displayName} requires Snapdragon Spaces SDK.", "OK");
                    return;
                }

                var coreSamplesParam = Report.GetRequiredAssetInfo(Validator.SpacesPackageName, "Core Samples");
                if (coreSamplesParam is not { Exists: true })
                {
                    EditorUtility.DisplayDialog("Error", $"{displayName} requires Snapdragon Spaces Core Samples.", "OK");
                    return;
                }

                var fusionSamplesParam = Report.GetRequiredAssetInfo(Validator.SpacesPackageName, "Fusion Samples");
                if (fusionSamplesParam is not { Exists: true })
                {
                    EditorUtility.DisplayDialog("Error", $"{displayName} requires Snapdragon Spaces Fusion Samples.", "OK");
                    return;
                }

                if (!ConfirmSampleOverride(displayName, param)) return;
                SampleImporter.ImportSample(displayName, param);

                EditorUtility.DisplayDialog("Success", $"{displayName} imported successfully.", "OK");
                Window.Validate();
            });

            AddButtonClickListener("ImportMirzaLibraryButton", () =>
            {
                var path = Path.GetFullPath(AssetDatabase.GetAssetPath(Window.MirzaLibraryAsset));
                //Debug.Log("Source Path: " + path);
                var copyPath = Path.Combine(Application.dataPath, "../Packages/mirza_library.tgz");
                File.Copy(path, copyPath, true);
                if (!File.Exists(copyPath))
                {
                    EditorUtility.DisplayDialog("Error", "Failed to copy MiRZA Library.", "OK");
                }

                _addMirzaLibraryRequest = Client.Add("file:" + "../Packages/mirza_library.tgz");
                EditorApplication.update += CheckAddMirzaLibraryRequest;
            });

            AddButtonClickListener("SetMinimumAPILevelButton", () =>
            {
                try
                {
                    PlayerSettings.Android.minSdkVersion = (AndroidSdkVersions)33;
                    if ((int)PlayerSettings.Android.targetSdkVersion < 33 &&
                        (int)PlayerSettings.Android.targetSdkVersion != 0)
                    {
                        PlayerSettings.Android.targetSdkVersion = (AndroidSdkVersions)33;
                    }
                }
                catch (Exception)
                {
                    EditorUtility.DisplayDialog("Error", "Failed to Set Android Minimum API Level to 33.", "OK");
                    return;
                }

                Window.Validate();
            });

            AddOpenLinkButtonClickListener("OpenMirzaLibraryGuideButton", "setup/mirza-sdk/mirza-library/overview");

            AddButtonClickListener("SwitchToAndroidButton", () =>
            {
                if (EditorUserBuildSettings.activeBuildTarget == BuildTarget.Android)
                {
                    EditorUtility.DisplayDialog("Error", "Already switched to Android.", "OK");
                    return;
                }

                EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.Android, BuildTarget.Android);

                Window.Validate();
            });

            AddButtonClickListener("OpenSceneSetupButton", () => { Window.MenuListView.selectedIndex = 1; });

            AddButtonClickListener("EnableAllowUnsafeCodeButton", () =>
            {
                var param = ValidationUtility.GetAllowUnsafeCode();
                if (!param.Exists)
                {
                    EditorUtility.DisplayDialog("Error", "Failed to get Allow Unsafe Code.", "OK");
                    return;
                }

                try
                {
                    Utility.SetAllowUnsafeCode(true);
                }
                catch (Exception e)
                {
                    Debug.LogError("Failed to enable Allow Unsafe Code: " + e.Message);
                    EditorUtility.DisplayDialog("Error", "Failed to enable Allow Unsafe Code.", "OK");
                    return;
                }

                Window.Validate();
                CompilationPipeline.RequestScriptCompilation();
            });

            AddButtonClickListener("DisableInitializeXROnStartupButton", () =>
            {
                try
                {
                    Utility.SetInitializeXrOnStartup(BuildTargetGroup.Android, false);
                }
                catch (Exception)
                {
                    EditorUtility.DisplayDialog("Error", "Failed to Disable Initialize XR on Startup.\n"
                                                         + "Please switch to the Android tab and Disable manually Initialize XR on Startup.",
                        "OK");
                    SettingsService.OpenProjectSettings("Project/XR Plug-in Management");
                    return;
                }

                Window.Validate();
            });

            AddButtonClickListener("ChangeOpenXRVersionButton", () =>
            {
                _addOpenXRPluginRequest = Client.Add("com.unity.xr.openxr@1.10.0");
                EditorApplication.update += CheckAddOpenXRRequest;
            });

            AddButtonClickListener("OpenPlayerSettingsButton", () =>
            {
                SettingsService.OpenProjectSettings("Project/Player");
            });

            AddButtonClickListener("OpenXRPluginManagementButton", () =>
            {
                SettingsService.OpenProjectSettings("Project/XR Plug-in Management");
            });

            AddButtonClickListener("OpenQCHTSamplesButton", () =>
            {
                var info = Report.GetRequiredPackageInfo(Validator.InteractionsPackageName);
                if (info is not { Exists: true })
                {
                    EditorUtility.DisplayDialog("Error", "QCHT Unity Interactions package not found.", "OK");
                    return;
                }
                UnityEditor.PackageManager.UI.Window.Open(Validator.InteractionsPackageName);
                Utility.OpenSamplesTabInPackageManagerWindow();
            });

#if UNITY_2023_1_OR_NEWER
            PanelParent.Q<VisualElement>("ChangeApplicationEntry").style.display = DisplayStyle.Flex;

                AddButtonClickListener("ChangeToActivityButton", () =>
                {
                    PlayerSettings.Android.applicationEntry = AndroidApplicationEntry.Activity;
                    Window.Validate();
                });
#else
            PanelParent.Q<VisualElement>("ChangeApplicationEntry").style.display = DisplayStyle.None;
#endif
        }

        public override void RefreshPanel()
        {
            var isImportedMirzaLibrary = Report.GetRequiredPackageInfo(Validator.MirzaLibraryPackageName).Exists;
            var recommendedAndroidMinimumApiLevel = isImportedMirzaLibrary ? 33 : 29;
            var recommendedAndroidTargetApiLevel = isImportedMirzaLibrary ? 33 : 31;

            UpdateValueLabel("SpacesSdkVersionLabel", Report.GetRequiredPackageInfo(Validator.SpacesPackageName),
                true);

            UpdateValueLabel("SpacesSdkCoreSamplesLabel",
                Report.GetRequiredAssetInfo(Validator.SpacesPackageName, "Core Samples"));

            UpdateValueLabel("SpacesSdkFusionSamplesLabel",
                Report.GetRequiredAssetInfo(Validator.SpacesPackageName, "Fusion Samples"));

            UpdateValueLabel("QchtSdkVersionLabel", Report.GetRequiredPackageInfo(Validator.InteractionsPackageName));

            UpdateValueLabel("QchtSdkCoreAssetsLabel",
                Report.GetRequiredAssetInfo(Validator.InteractionsPackageName, "Core Assets"));

            UpdateValueLabel("QchtSdkQchtSamplesLabel",
                Report.GetRequiredAssetInfo(Validator.InteractionsPackageName, "QCHT Samples"));

            UpdateValueLabel("MirzaLibraryVersionLabel",
                Report.GetRequiredPackageInfo(Validator.MirzaLibraryPackageName));

            UpdateOpenXRVersionLabel("OpenXRPluginVersionLabel",
                Report.GetRequiredPackageInfo(Validator.OpenXRPluginPackageName),
                Report.ProjectParams.IgnoreOpenXRVersion);

            UpdateValueLabelByName("InitializeXROnStartupLabel", Report.ProjectParams.InitializeXROnStartup, true);

            UpdateValueLabelByName("OpenXRPluginEnabledLabel", Report.ProjectParams.OpenXRPluginEnabled, true);

            UpdateValueLabelByName("SnapdragonSpacesFeatureGroupEnabledLabel",
                Report.ProjectParams.SnapdragonSpacesFeatureGroupEnabled, true);

            UpdateValueLabelByName("BaseRuntimeFeatureEnabledLabel", Report.ProjectParams.BaseRuntimeFeatureEnabled,
                true);

            UpdateValueLabelByName("LaunchAppOnViewerLabel", Report.ProjectParams.LaunchAppOnViewer, true);

            UpdateValueLabelByName("LaunchControllerOnHostLabel", Report.ProjectParams.LaunchControllerOnHost, true);

            UpdateValueLabelByName("ExportHeadlessLabel", Report.ProjectParams.ExportHeadless, true);

            UpdateValueLabelByName("DualRenderFusionFeatureEnabledLabel",
                Report.ProjectParams.DualRenderFusionFeatureEnabled, true);

            UpdateValueLabelByName("SnapdragonSpacesExperimentalFeatureGroupEnabledLabel",
                Report.ProjectParams.SnapdragonSpacesExperimentalFeatureGroupEnabled);

            UpdateValueLabelByName("UnityVersionLabel", Report.ProjectParams.UnityVersion, true);

            UpdateValueLabelByName("PlatformLabel", Report.ProjectParams.Platform, true);
            // EditorUserBuildSettings.SwitchActiveBuildTarget(BuildTargetGroup.Android, BuildTarget.Android);

            UpdateValueLabelByName("ActiveInputHandlingLabel", Report.ProjectParams.ActiveInputHandling, true);

            UpdateNameLabel("AndroidMinimumApiLevelLabel", $"Android Minimum API Level (Recommended Version : {recommendedAndroidMinimumApiLevel} or higher)");
            UpdateValueLabelByName("AndroidMinimumApiLevelLabel", Report.ProjectParams.AndroidMinimumApiLevel, true);

            UpdateNameLabel("AndroidTargetApiLevelLabel", $"Android Target API Level (Recommended Version : {recommendedAndroidTargetApiLevel} or higher)");
            UpdateValueLabelByName("AndroidTargetApiLevelLabel", Report.ProjectParams.AndroidTargetApiLevel, true);

            RefleshControllerImportButton(PanelParent.Q<Button>("ImportControllerButton"), Report.GetRequiredPackageInfo(Validator.ConfigToolPackageName));

            PanelParent.Q<Button>("SetMinimumAPILevelButton")
                .SetEnabled(!(Report.ProjectParams.AndroidMinimumApiLevel.Actual == "AndroidApiLevel33"));

            bool isAllowedUnsafeCode = Report.ProjectParams.AllowUnsafeCode.Exists &&
                                       Report.ProjectParams.AllowUnsafeCode.Actual == "True";
            PanelParent.Q<Button>("EnableAllowUnsafeCodeButton").text =
                isAllowedUnsafeCode ? "Enabled" : "Enable";
            PanelParent.Q<Button>("EnableAllowUnsafeCodeButton").SetEnabled(!isAllowedUnsafeCode);

            bool disabledInitializeXROnStartup = Report.ProjectParams.InitializeXROnStartup.Exists &&
                                                 Report.ProjectParams.InitializeXROnStartup.Actual == false.ToString();
            PanelParent.Q<Button>("DisableInitializeXROnStartupButton").text =
                disabledInitializeXROnStartup ? "Disabled" : "Disable";
            PanelParent.Q<Button>("DisableInitializeXROnStartupButton").SetEnabled(!disabledInitializeXROnStartup);

            bool changedOpenXRVersion = Report.GetRequiredPackageInfo(Validator.OpenXRPluginPackageName).Exists &&
                                        Report.GetRequiredPackageInfo(Validator.OpenXRPluginPackageName).ActualVersion == "1.10.0";
            PanelParent.Q<Button>("ChangeOpenXRVersionButton").text =
                changedOpenXRVersion ? "Changed to 1.10.0" : "Change to 1.10.0";
            PanelParent.Q<Button>("ChangeOpenXRVersionButton").SetEnabled(!changedOpenXRVersion);

            bool isAndroidPlatform = Report.ProjectParams.Platform.Exists &&
                                     Report.ProjectParams.Platform.Actual == "Android";
            PanelParent.Q<Button>("SwitchToAndroidButton").SetEnabled(!isAndroidPlatform);
            PanelParent.Q<Button>("SwitchToAndroidButton").text = isAndroidPlatform ? "Switched to Android" : "Switch to Android";

#if UNITY_2023_1_OR_NEWER
            {
                bool isActivity = PlayerSettings.Android.applicationEntry == AndroidApplicationEntry.Activity;
                PanelParent.Q<Button>("ChangeToActivityButton").SetEnabled(!isActivity);
                PanelParent.Q<Button>("ChangeToActivityButton").text = isActivity ? "Changed to Activity" : "Change to Activity";
            }
#endif
        }

        private void UpdateNameLabel(string id, string name)
        {
            var label = PanelParent.Q<VisualElement>(id).Q<Label>("NameLabel");
            if (label == null) return;
            if (string.IsNullOrEmpty(name)) return;
            label.text = name;
        }

        private void CheckAddMirzaLibraryRequest()
        {
            if (_addMirzaLibraryRequest == null)
            {
                EditorApplication.update -= CheckAddMirzaLibraryRequest;
                return;
            }

            if (_addMirzaLibraryRequest.IsCompleted)
            {
                if (_addMirzaLibraryRequest.Status == StatusCode.Success)
                {
                    EditorUtility.DisplayDialog("Success", "MiRZA Library imported successfully.", "OK");
                }
                else
                {
                    EditorUtility.DisplayDialog("Error", "Failed to import MiRZA Library.", "OK");
                    Debug.Log(_addMirzaLibraryRequest.Error.message);
                }
                _addMirzaLibraryRequest = null;
            }
        }

        private void CheckAddOpenXRRequest()
        {
            if (_addOpenXRPluginRequest == null)
            {
                EditorApplication.update -= CheckAddOpenXRRequest;
                return;
            }

            if (_addOpenXRPluginRequest.IsCompleted)
            {
                if (_addOpenXRPluginRequest.Status == StatusCode.Success)
                {
                    EditorUtility.DisplayDialog("Success", "OpenXR Plugin version changed to 1.10.0.", "OK");
                }
                else
                {
                    EditorUtility.DisplayDialog("Error", "Failed to change OpenXR Plugin version to 1.10.0.", "OK");
                    Debug.Log(_addOpenXRPluginRequest.Error.message);
                }
                _addOpenXRPluginRequest = null;
            }
        }

        public void UpdateOpenXRVersionLabel(string id, PackageParam openXRParam, Param IgnoreOpenXRVersion)
        {
            bool ignoreOpenXRVersion = IgnoreOpenXRVersion.Exists && IgnoreOpenXRVersion.Actual == "True";
            bool required = true;
            var valueLabel = PanelParent.Q<VisualElement>(id).Q<Label>("ValueLabel");

            if (ignoreOpenXRVersion)
            {
                UpdateNameLabel(id, OpenXRTitle + OpenXRIgnoreVersionStatus);
            }
            else
            {
                UpdateNameLabel(id, OpenXRTitle + OpenXRRecommendedVersion);
            }

            if (valueLabel == null)
            {
                return;
            }

            valueLabel.tooltip = null;

            if (openXRParam == null)
            {
                valueLabel.Set(NotFoundLabelText, required ? ErrorValueLabelColor : WarningValueLabelColor);
                return;
            }

            if (openXRParam.Exists)
            {
                if (string.IsNullOrEmpty(openXRParam.MinRequiredVersion))
                {
                    valueLabel.Set(openXRParam.ActualVersion, DefaultValueLabelColor);
                }
                else
                {
                    if (string.Compare(openXRParam.ActualVersion, openXRParam.MinRequiredVersion, StringComparison.Ordinal) >= 0)
                    {
                        if (string.IsNullOrEmpty(openXRParam.ExpectedVersion))
                        {
                            valueLabel.Set(openXRParam.ActualVersion, DefaultValueLabelColor);
                        }
                        else
                        {
                            if (openXRParam.ActualVersion.Equals(openXRParam.ExpectedVersion))
                            {
                                valueLabel.Set(openXRParam.ActualVersion, DefaultValueLabelColor);
                            }
                            else
                            {
                                if (!ignoreOpenXRVersion)
                                {
                                    valueLabel.Set(openXRParam.ActualVersion, required ? ErrorValueLabelColor : WarningValueLabelColor, $"RequiredVersion: {openXRParam.ExpectedVersion}");
                                }
                                else
                                {
                                    valueLabel.Set(openXRParam.ActualVersion, DefaultValueLabelColor, $"RequiredVersion: {openXRParam.ExpectedVersion}");
                                }
                            }
                        }
                    }
                    else
                    {
                        valueLabel.Set(openXRParam.ActualVersion, required ? ErrorValueLabelColor : WarningValueLabelColor, $"MinRequiredVersion: {openXRParam.MinRequiredVersion}");
                    }
                }
            }
            else
            {
                valueLabel.Set(NotFoundLabelText, required ? ErrorValueLabelColor : WarningValueLabelColor);
            }
        }

        public void RefleshControllerImportButton(Button button, PackageParam param)
        {
            var displayName = "Dual Render Fusion Controller";
            if (button == null)
            {
                return;
            }

            var importStatusText = "Import";
            if (SampleImporter.IsSampleImported(displayName, param, out var isCurrentVersion))
            {
                importStatusText = isCurrentVersion ? "Reimport" : "Update";
            }
            button.text = $"{importStatusText} {displayName}";
        }
    }
}