﻿using System;
using System.Collections.Generic;
using UnityEditor;
using UnityEngine;
using UnityEngine.UIElements;
using Button = UnityEngine.UIElements.Button;
using Object = UnityEngine.Object;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    [InitializeOnLoad]
    public static class UnityConfigurationTool
    {
        private static readonly string PresentedKey = Application.dataPath + "MiRZA_Config_Tool_Presented";
        private static readonly string CategoryIndexKey = Application.dataPath + "MiRZA_Config_Tool_Category_Index";

        private static readonly string WindowWidthKey = Application.dataPath + "MiRZA_Config_Tool_Window_Width";
        private static readonly string WindowHeightKey = Application.dataPath + "MiRZA_Config_Tool_Window_Height";

        static UnityConfigurationTool()
        {
            EditorApplication.update += OnInit;
        }

        private static bool ConfigToolHasBeenPresented
        {
            get => EditorPrefs.GetBool(PresentedKey, false);
            set => EditorPrefs.SetBool(PresentedKey, value);
        }

        public static int WindowWidth
        {
            get => Mathf.Max(EditorPrefs.GetInt(WindowWidthKey, ConfigurationWindow.DefaultWindowSize.x), ConfigurationWindow.MinWindowSize.x);
            set => EditorPrefs.SetInt(WindowWidthKey, value);
        }

        public static int WindowHeight
        {
            get => Mathf.Max(EditorPrefs.GetInt(WindowHeightKey, ConfigurationWindow.DefaultWindowSize.y), ConfigurationWindow.MinWindowSize.y);
            set => EditorPrefs.SetInt(WindowHeightKey, value);
        }

        private static void OnInit()
        {
            EditorApplication.update -= OnInit;
            if (!EditorApplication.isPlayingOrWillChangePlaymode && !ConfigToolHasBeenPresented)
            {
                ConfigurationWindow.ShowWindow();
                ConfigToolHasBeenPresented = true;
            }
        }

        public static void SetCategoryIndex(int index)
        {
            EditorPrefs.SetInt(CategoryIndexKey, index);
        }

        public static int GetCategoryIndex()
        {
            return EditorPrefs.GetInt(CategoryIndexKey, 0);
        }

        public static void CleanEditorPrefsData()
        {
            EditorPrefs.DeleteKey(PresentedKey);
            EditorPrefs.DeleteKey(CategoryIndexKey);
            EditorPrefs.DeleteKey(WindowWidthKey);
            EditorPrefs.DeleteKey(WindowHeightKey);
            Debug.Log("EditorPrefs data added by MiRZA Configuration Tool is deleted.");
        }
    }

    public class ConfigurationWindow : EditorWindow
    {
        public static readonly Vector2Int DefaultWindowSize = new(950, 800);
        public static readonly Vector2Int MinWindowSize = new(300, 200);

        public enum PanelType
        {
            ProjectSetup,
            SceneSetup,
            Simulator,
            Templates,
            Samples,
            Debug,
            Other
        }

        [SerializeField] private VisualTreeAsset _configurationWindowAsset = default;

        [SerializeField] private VisualTreeAsset _cameraListItemAsset = default;

        [SerializeField] private VisualTreeAsset _menuListItemAsset = default;

        [SerializeField] private VisualTreeAsset _projectSetupPanelAsset = default;

        [SerializeField] private VisualTreeAsset _sceneSetupPanelAsset = default;

        [SerializeField] private VisualTreeAsset _simulatorPanelAsset = default;

        [SerializeField] private VisualTreeAsset _templatesPanelAsset = default;

        [SerializeField] private VisualTreeAsset _samplesPanelAsset = default;

        [SerializeField] private VisualTreeAsset _debugPanelAsset = default;

        [SerializeField] private VisualTreeAsset _otherPanelAsset = default;

        [SerializeField] private Object _mirzaLibraryAsset;

        [SerializeField] private Object _inGameDebugConsoleAsset;

        private ScrollView _contentScrollView;

        private ValidationReport _report;

        private PanelType _currentPanelType;

        private Button _refreshButton;

        private ListView _menuListView;

        private bool _isDualRenderFusionMode = true;

        private string _currentVersion = string.Empty;

        private ProjectSetupPanel _projectSetupPanel;
        private SceneSetupPanel _sceneSetupPanel;
        private SimulatorPanel _simulatorPanel;
        private TemplatesPanel _templatesPanel;
        private SamplesPanel _samplesPanel;
        private DebugPanel _debugPanel;
        private OtherPanel _otherPanel;

        private static readonly string WindowTitle = "MiRZA Configuration Tool";

        public ValidationReport Report => _report;

        public ScrollView PanelParent => _contentScrollView;

        public Object MirzaLibraryAsset => _mirzaLibraryAsset;

        public Object InGameDebugConsoleAsset => _inGameDebugConsoleAsset;

        public ListView MenuListView => _menuListView;

        public VisualTreeAsset CameraListItemAsset => _cameraListItemAsset;

        public bool IsDualRenderFusionMode => _isDualRenderFusionMode;

        private Vector2 _prevSize;

        private VisualElement _windowElement;

        [MenuItem("Window/XR/MiRZA/Configuration Tool")]
        public static void ShowWindow()
        {
            var window = GetWindow<ConfigurationWindow>();
            window.titleContent = new GUIContent(WindowTitle);

            var width = UnityConfigurationTool.WindowWidth;
            var height = UnityConfigurationTool.WindowHeight;
            var res = Screen.currentResolution;
            var x = (res.width - width) / 2f;
            var y = (res.height - height) / 2f;
            window.position = new Rect(x, y, width, height);
            window.minSize = new Vector2(MinWindowSize.x, MinWindowSize.y);
            UnityConfigurationTool.SetCategoryIndex(0);
        }

        private void CreateGUI()
        {
            var root = rootVisualElement;

            var container = _configurationWindowAsset.Instantiate();
            root.Add(container);

            var menuSource = new[]
            {
                new MenuItemData("project_setup", "Project Setup"),
                new MenuItemData("scene_setup", "Scene Setup"),
                new MenuItemData("simulator", "Simulator"),
                new MenuItemData("templates", "Templates"),
                new MenuItemData("samples", "Samples"),
                new MenuItemData("debug", "Debug"),
                new MenuItemData("other", "Other"),
            };

            _menuListView = root.Q<ListView>("MenuListView");
            _menuListView.makeItem = () =>
            {
                var item = _menuListItemAsset.Instantiate();
                return item;
            };
            _menuListView.bindItem = (element, i) =>
            {
                var menuLabel = element.Q<Label>("MenuLabel");
                menuLabel.text = menuSource[i].DisplayName;
                //(element as Label).text = menuSource[i];
            };
            _menuListView.itemsSource = menuSource;
            _menuListView.selectionType = SelectionType.Single;
            _menuListView.selectionChanged += OnSelectionChanged;

            _contentScrollView = root.Q<ScrollView>("ContentScrollView");

            _menuListView.selectedIndex = UnityConfigurationTool.GetCategoryIndex();

            _refreshButton = root.Q<Button>("RefreshButton");
            _refreshButton.clickable.clicked += OnRefreshButtonClicked;

            // Fit content size
            _windowElement = rootVisualElement.Q<VisualElement>("Window");
            _prevSize = position.size;
            _windowElement.style.minWidth = _prevSize.x;
            _windowElement.style.minHeight = _prevSize.y;

            Validate();
        }

        private void OnEnable()
        {
            _prevSize = position.size;
        }

        private void OnGUI()
        {
            if (position.size != _prevSize)
            {
                _prevSize = position.size;

                if (_windowElement == null)
                {
                    _windowElement = rootVisualElement.Q<VisualElement>("Window");
                }

                if(_windowElement != null)
                {
                    _windowElement.style.minWidth = _prevSize.x;
                    _windowElement.style.minHeight = _prevSize.y;
                    UnityConfigurationTool.WindowWidth = (int)_prevSize.x;
                    UnityConfigurationTool.WindowHeight = (int)_prevSize.y;
                }
            }
        }

        private void OnRefreshButtonClicked()
        {
            Validate();
        }

        public void Validate()
        {
            var refreshStatusLabel = rootVisualElement.Q<Label>("RefreshStatusLabel");
            refreshStatusLabel.text = "Loading...";

            if (_report != null)
            {
                _currentVersion = _report.GetRequiredPackageInfo(Validator.ConfigToolPackageName)?.ActualVersion;
            }

            var validator = new Validator(_isDualRenderFusionMode);
            validator.Validate(OnValidationComplete);
            RefreshPanel(_currentPanelType, false);
        }

        private void OnValidationComplete(ValidationReport obj)
        {
            _report = obj;
            CheckConfigurationToolVersion();
            RefreshPanel(_currentPanelType, true);
        }

        private void OnSelectionChanged(IEnumerable<object> arr)
        {
            UnityConfigurationTool.SetCategoryIndex(_menuListView.selectedIndex);
            foreach (var item in arr)
            {
                var menuItemData = item as MenuItemData;
                if (menuItemData != null)
                {
                    SelectPanel(menuItemData.ID);
                }
            }
        }

        private void SelectPanel(string id)
        {
            _contentScrollView.Clear();
            switch (id)
            {
                case "project_setup":
                    _currentPanelType = PanelType.ProjectSetup;
                    break;
                case "scene_setup":
                    _currentPanelType = PanelType.SceneSetup;
                    break;
                case "simulator":
                    _currentPanelType = PanelType.Simulator;
                    break;
                case "templates":
                    _currentPanelType = PanelType.Templates;
                    break;
                case "samples":
                    _currentPanelType = PanelType.Samples;
                    break;
                case "debug":
                    _currentPanelType = PanelType.Debug;
                    break;
                case "other":
                    _currentPanelType = PanelType.Other;
                    break;
            }
            ShowPanel(_currentPanelType);
        }

        private void ShowPanel(PanelType panelType)
        {
            switch (panelType)
            {
                case PanelType.ProjectSetup:
                    {
                        _projectSetupPanel = new ProjectSetupPanel(_projectSetupPanelAsset, this);
                        break;
                    }
                case PanelType.SceneSetup:
                    {
                        _sceneSetupPanel = new SceneSetupPanel(_sceneSetupPanelAsset, this);
                        break;
                    }
                case PanelType.Simulator:
                    {
                        _simulatorPanel = new SimulatorPanel(_simulatorPanelAsset, this);
                        break;
                    }
                case PanelType.Templates:
                    {
                        _templatesPanel = new TemplatesPanel(_templatesPanelAsset, this);
                        break;
                    }
                case PanelType.Samples:
                    {
                        _samplesPanel = new SamplesPanel(_samplesPanelAsset, this);
                        break;
                    }
                case PanelType.Debug:
                    {
                        _debugPanel = new DebugPanel(_debugPanelAsset, this);
                        break;
                    }
                case PanelType.Other:
                    {
                        _otherPanel = new OtherPanel(_otherPanelAsset, this);
                        break;
                    }
                default:
                    throw new ArgumentOutOfRangeException(nameof(panelType), panelType, null);
            }

            // Reset scroll position
            if (_contentScrollView != null)
            {
                _contentScrollView.scrollOffset = Vector2.zero;
            }

            Validate();
        }

        private void RefreshPanel(PanelType panelType, bool updateStatus)
        {
            if (_report == null)
            {
                return;
            }

            switch (panelType)
            {
                case PanelType.ProjectSetup:
                    {
                        _projectSetupPanel.RefreshPanel();
                        break;
                    }
                case PanelType.SceneSetup:
                    {
                        _sceneSetupPanel.RefreshPanel();
                        break;
                    }
                case PanelType.Simulator:
                    {
                        _simulatorPanel.RefreshPanel();
                        break;
                    }
                case PanelType.Templates:
                    {
                        _templatesPanel.RefreshPanel();
                        break;
                    }
                case PanelType.Samples:
                    {
                        _samplesPanel.RefreshPanel();
                        break;
                    }
                case PanelType.Debug:
                    {
                        _debugPanel.RefreshPanel();
                        break;
                    }
                case PanelType.Other:
                    {
                        _otherPanel.RefreshPanel();
                        break;
                    }
                default:
                    throw new ArgumentOutOfRangeException(nameof(panelType), panelType, null);
            }

            if (updateStatus)
            {
                var refreshStatusLabel = rootVisualElement.Q<Label>("RefreshStatusLabel");
                refreshStatusLabel.text = $"Last updated: {_report.CreateDate}";
            }

            var toolVersionLabel = rootVisualElement.Q<Label>("ToolVersionLabel");
            var toolInfo = _report.GetRequiredPackageInfo(Validator.ConfigToolPackageName);
            toolVersionLabel.text = toolInfo != null ? toolInfo.ActualVersion : "Not Found";
        }

        private void CloseWindow()
        {
            var windows = Resources.FindObjectsOfTypeAll<EditorWindow>();
            foreach (var window in windows)
            {
                if (window.titleContent.text == WindowTitle)
                {
                    window.Close();
                }
            }
        }

        private void CheckConfigurationToolVersion()
        {
            var info = _report.GetRequiredPackageInfo(Validator.ConfigToolPackageName);
            if (info != null)
            {
                if (string.IsNullOrEmpty(_currentVersion))
                {
                    _currentVersion = info.ActualVersion;
                }
                else if (_currentVersion != info.ActualVersion)
                {
                    _currentVersion = info.ActualVersion;
                    CloseWindow();
                }
            }
        }
    }
}