﻿using System;
using System.Reflection;
using Unity.XR.CoreUtils;
using UnityEditor;
using UnityEditor.PackageManager.UI;
using UnityEngine;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    /// <summary>
    /// 新規 GameView を作成し、XROrigin以下のCameraと合わせてMiRZAのシミュレーション用の設定を行う
    /// </summary>
    public class CameraSimulator : MonoBehaviour
    {
        private static readonly float Fov = 45f;
        private static readonly float DisplayWidth = 1920f;
        private static readonly float DisplayHeight = 1080f;

        public static void ConfigureSimulator()
        {
            CreateGameView();
            ConfigureXRCamera();
        }

        /// <summary>
        /// 新規 GameView を作成し、TargetDisplay=2 と 描画解像度を設定する
        /// </summary>
        private static void CreateGameView()
        {
            var gameViewType = typeof(UnityEditor.Editor).Assembly.GetType("UnityEditor.GameView");

            // GameView のメソッド SetCustomResolution(Vector2, string) をリフレクションで取得
            var setCustomResolutionMethod = gameViewType.GetMethod(
                "SetCustomResolution",
                BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance,
                null,
                new[] { typeof(Vector2), typeof(string) },
                null
            );

            if (setCustomResolutionMethod == null)
            {
                Debug.LogError("SetCustomResolution method not found. It might be internal or changed in this Unity version.");
                return;
            }

            var newGameView = ScriptableObject.CreateInstance(gameViewType) as EditorWindow;
            if (newGameView != null)
            {
                // m_TargetDisplay を 2 に設定
                var so = new SerializedObject(newGameView);
                var displayProp = so.FindProperty("m_TargetDisplay");
                displayProp.intValue = 1;
                so.ApplyModifiedProperties();
                // カスタム解像度"MiRZA(1920x1080)"などが存在する場合はそれを利用し、存在しない場合は新規作成して設定する。
                setCustomResolutionMethod.Invoke(newGameView, new object[] { new Vector2(DisplayWidth, DisplayHeight), $"MiRZA({DisplayWidth}x{DisplayHeight})" });

                // 追加した GameView を表示
                newGameView.Show();
                newGameView.Focus();
            }
            Debug.Log("Complete Creating MiRZA GameView.");
        }

        /// <summary>
        /// XROrigin以下のCameraをMiRZA GameView用に設定する
        /// </summary>
        private static void ConfigureXRCamera()
        {
            // ヒエラルキー内のXR Originコンポーネントにアタッチされているカメラを取得
            XROrigin xrOrigin = null;

            // シーン内のすべてのルートオブジェクトを取得
            foreach (GameObject root in UnityEngine.SceneManagement.SceneManager.GetActiveScene().GetRootGameObjects())
            {
                // 子オブジェクト、非アクティブを含めて検索
                xrOrigin = root.GetComponentInChildren<XROrigin>(true);
                if (xrOrigin != null)
                    break;
            }

            if (xrOrigin == null)
            {
                Debug.LogError("XROrigin not found in the scene.");
                return;
            }
            var targetCamera = xrOrigin.Camera;
            if (targetCamera == null)
            {
                Debug.LogError("Camera not found in the XROrigin.");
                return;
            }
            if (!targetCamera.enabled)
            {
                Debug.LogWarning("Camera is disabled.");
            }
            if (!targetCamera.gameObject.activeInHierarchy)
            {
                Debug.LogWarning("Camera Object or parent is not active.");
            }

            // カメラのターゲットディスプレイを設定
            targetCamera.targetDisplay = 1;
            // カメラのField Of View Axisを設定
            float fovVertical = Camera.HorizontalToVerticalFieldOfView(Fov, DisplayWidth / DisplayHeight);
            targetCamera.fieldOfView = fovVertical;
            Debug.Log("Complete XROrigin Camera settings for Mirza.");

            // ヒエラルキー内のカメラを選択状態にする
            Selection.activeGameObject = targetCamera.gameObject;
        }
    }
}