﻿using System;
using UnityEditor;
using UnityEditor.SceneManagement;
using UnityEngine;
using UnityEngine.UIElements;
using Object = UnityEngine.Object;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public class BasePanel
    {
        public static readonly Color DefaultValueLabelColor = Color.white;

        public static readonly Color WarningValueLabelColor = Color.yellow;

        public static readonly Color ErrorValueLabelColor = Color.red;

        public static readonly string NotFoundLabelText = "Not found";

        public ConfigurationWindow Window { get; set; }

        public ScrollView PanelParent => Window.PanelParent;

        public ValidationReport Report => Window.Report;

        public BasePanel(VisualTreeAsset visualTreeAsset, ConfigurationWindow window)
        {
            Window = window;
            var panel = visualTreeAsset.Instantiate();
            PanelParent.Add(panel);
        }

        public virtual void RefreshPanel()
        {
        }

        public void UpdateValueLabelByName(string id, Param info, bool required = false)
        {
            var label = PanelParent.Q<VisualElement>(id).Q<Label>("ValueLabel");

            UpdateValueLabel(label, info, required);
        }

        public void UpdateValueLabel(Label label, Param info, bool required = false)
        {
            if (label == null)
            {
                return;
            }

            label.tooltip = null;

            if (info == null)
            {
                label.Set(NotFoundLabelText, required ? ErrorValueLabelColor : WarningValueLabelColor);
                return;
            }

            label.tooltip = info.Message;

            if (info.Exists)
            {
                if (string.IsNullOrEmpty(info.Expected))
                {
                    label.Set(info.Actual, DefaultValueLabelColor);
                }
                else
                {
                    if (info.Actual == info.Expected)
                    {
                        label.Set(info.Actual, DefaultValueLabelColor);
                    }
                    else
                    {
                        label.Set(info.Actual, required ? ErrorValueLabelColor : WarningValueLabelColor);
                    }
                }
            }
            else
            {
                label.Set(NotFoundLabelText, required ? ErrorValueLabelColor : WarningValueLabelColor);
            }
        }

        public void UpdateValueLabel(string id, PackageParam param, bool required = false)
        {
            var label = PanelParent.Q<VisualElement>(id).Q<Label>("ValueLabel");

            if (label == null)
            {
                return;
            }

            label.tooltip = null;

            if (param == null)
            {
                label.Set(NotFoundLabelText, required ? ErrorValueLabelColor : WarningValueLabelColor);
                return;
            }

            if (param.Exists)
            {
                if (string.IsNullOrEmpty(param.MinRequiredVersion))
                {
                    label.Set(param.ActualVersion, DefaultValueLabelColor);
                }
                else
                {
                    if (string.Compare(param.ActualVersion, param.MinRequiredVersion, StringComparison.Ordinal) >= 0)
                    {
                        if (string.IsNullOrEmpty(param.ExpectedVersion))
                        {
                            label.Set(param.ActualVersion, DefaultValueLabelColor);
                        }
                        else
                        {
                            if (param.ActualVersion.Equals(param.ExpectedVersion))
                            {
                                label.Set(param.ActualVersion, DefaultValueLabelColor);
                            }
                            else
                            {
                                label.Set(param.ActualVersion, required ? ErrorValueLabelColor : WarningValueLabelColor,  $"RequiredVersion: {param.ExpectedVersion}");
                            }
                        }
                    }
                    else
                    {
                        label.Set(param.ActualVersion, required ? ErrorValueLabelColor : WarningValueLabelColor,  $"MinRequiredVersion: {param.MinRequiredVersion}");
                    }
                }
            }
            else
            {
                label.Set(NotFoundLabelText, required ? ErrorValueLabelColor : WarningValueLabelColor);
            }
        }

        public void UpdateValueLabel(string id, AssetParam param, bool required = false)
        {
            var label = PanelParent.Q<VisualElement>(id).Q<Label>("ValueLabel");

            if (label == null)
            {
                return;
            }

            label.tooltip = null;

            if (param == null)
            {
                label.Set(NotFoundLabelText, required ? ErrorValueLabelColor : WarningValueLabelColor);
                return;
            }

            if (param.Exists)
            {
                label.Set("Installed", DefaultValueLabelColor);
            }
            else
            {
                label.Set(NotFoundLabelText, required ? ErrorValueLabelColor : WarningValueLabelColor);
            }
        }

        public void UpdateValueLabelExpectedNotFound(string id, Param info, bool required = false)
        {
            var label = PanelParent.Q<VisualElement>(id).Q<Label>("ValueLabel");

            if (label == null)
            {
                return;
            }

            label.tooltip = null;

            if (info == null)
            {
                label.Set(NotFoundLabelText, DefaultValueLabelColor);
                return;
            }

            label.tooltip = info.Message;

            if (info.Exists)
            {
                label.Set(info.Actual, required ? ErrorValueLabelColor : WarningValueLabelColor);
            }
            else
            {
                label.Set(NotFoundLabelText, DefaultValueLabelColor);
            }
        }

        public void UpdateValueLabelExpectedNotFoundOrDisabled(string id, Param info, bool required = false)
        {
            var label = PanelParent.Q<VisualElement>(id).Q<Label>("ValueLabel");

            if (label == null)
            {
                return;
            }

            label.tooltip = null;

            if (info == null)
            {
                label.Set(NotFoundLabelText, DefaultValueLabelColor);
                return;
            }

            label.tooltip = info.Message;

            if (info.Exists)
            {
                if (info.Actual.Contains("Disabled"))
                {
                    label.Set(info.Actual, DefaultValueLabelColor);
                    return;
                }
                label.Set(info.Actual, required ? ErrorValueLabelColor : WarningValueLabelColor);
            }
            else
            {
                label.Set(NotFoundLabelText, DefaultValueLabelColor);
            }
        }

        public void UpdateBoolValueLabel(string id, bool value, bool required = false)
        {
            var label = PanelParent.Q<VisualElement>(id).Q<Label>("ValueLabel");
            label.Set(value.ToString(), value ? DefaultValueLabelColor : ( required ? ErrorValueLabelColor : WarningValueLabelColor));
        }

        public void AddButtonClickListener(string id, Action onClick)
        {
            try
            {
                PanelParent.Q<Button>(id).clickable.clicked += onClick;
            }
            catch (Exception e)
            {
                Debug.LogError($"Failed to add button click listener: {id}, " + e.Message);
            }
        }

        public void AddOpenLinkButtonClickListener(string id, string page)
        {
            try
            {
                PanelParent.Q<Button>(id).clickable.clicked += () =>
                {
                    Application.OpenURL(Constants.GuidePageUrl + page);
                };
            }
            catch (Exception e)
            {
                Debug.LogError($"Failed to add button click listener: {id}, " + e.Message);
            }
        }

        public void AddSampleImportButtonClickListener(string id, Action onClick)
        {
            try
            {
                PanelParent.Q<VisualElement>(id).Q<Button>("ImportButton").clickable.clicked += onClick;
            }
            catch (Exception e)
            {
                Debug.LogError($"Failed to add button click listener: {id}, " + e.Message);
            }
        }

        public void RefleshSampleImportButton(Button button, string displayName, PackageParam param)
        {
            if (button == null)
            {
                return;
            }

            var isImported = SampleImporter.IsSampleImported(displayName, param, out var isCurrentVersion);
            if (isImported)
            {
                button.text = isCurrentVersion ? "Reimport" : "Update";
            }
            else
            {
                button.text = "Import";
            }
        }

        public bool ConfirmSampleOverride(string displayName, PackageParam param)
        {
            var isImported = SampleImporter.IsSampleImported(displayName, param, out var isCurrentVersion);
            if (isImported)
            {
                if (!isCurrentVersion)
                {
                    var message = $"A different version of the sample is already imported.\nIt will be deleted when you update.\nAre you sure you want to continue?";
                    return EditorUtility.DisplayDialog("MiRZA Configuration Tool", message, "OK", "Cancel");
                }
            }
            return true;
        }

        public void OpenAndSelectScene(string scenePath)
        {
            AssetDatabase.Refresh();
            EditorSceneManager.OpenScene(scenePath);
            Object sceneAsset = AssetDatabase.LoadAssetAtPath<Object>(scenePath);
            if (sceneAsset != null)
            {
                Selection.activeObject = sceneAsset;
                EditorGUIUtility.PingObject(sceneAsset);
            }
        }
    }
}