﻿using System;
using System.IO;
using System.Reflection;
using UnityEditor;
using UnityEditor.Events;
using UnityEngine;
using UnityEngine.Events;

namespace NTTQONOQ.Android.MiRZA.SDK.ConfigurationTool.Editor
{
    public static class AssemblyUtility
    {
        public static Assembly FindAssemblyByName(string assemblyName = "Assembly-CSharp")
        {
            var assemblies = AppDomain.CurrentDomain.GetAssemblies();

            foreach (var assembly in assemblies)
            {
                var name = assembly.GetName().Name;
                if (name == assemblyName)
                {
                    return assembly;
                }
            }

            return null;
        }

        public static void CallMethodFromAssembly(string assemblyName, string typeName, string methodName)
        {
            var targetAssembly = FindAssemblyByName(assemblyName);

            if (targetAssembly == null)
            {
                Debug.LogError("Assembly not found: " + assemblyName);
                return;
            }

            var targetType = targetAssembly.GetType(typeName);
            if (targetType == null)
            {
                Debug.LogError("Type not found: " + typeName);
                return;
            }

            var methodInfo = targetType.GetMethod(methodName, BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance | BindingFlags.Static);
            if (methodInfo == null)
            {
                Debug.LogError("Method notfound: " + methodName);
                return;
            }

            if (methodInfo.IsStatic)
            {
                methodInfo.Invoke(null, null);
            }
            else
            {
                var classInstance = Activator.CreateInstance(targetType, null);
                methodInfo.Invoke(classInstance, null);
            }
        }

        public static bool TryGetFieldValueFromAssembly(string assemblyName, string typeName, string fieldName, object target, out object result)
        {
            result = null;

            var targetAssembly = FindAssemblyByName(assemblyName);

            if (targetAssembly == null)
            {
                Debug.LogError("Assembly not found: " + assemblyName);
                return false;
            }

            var targetType = targetAssembly.GetType(typeName);
            if (targetType == null)
            {
                Debug.LogError("Type not found: " + typeName);
                return false;
            }

            var fieldInfo = targetType.GetField(fieldName, BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance | BindingFlags.Static);
            if (fieldInfo == null)
            {
                Debug.LogError("Field notfound: " + fieldName);
                return false;
            }

            result = fieldInfo.GetValue(target);
            return true;
        }

        public static bool TrySetFieldValueFromAssembly(string assemblyName, string typeName, string fieldName, object target, object value)
        {
            var targetAssembly = FindAssemblyByName(assemblyName);

            if (targetAssembly == null)
            {
                Debug.LogError("Assembly not found: " + assemblyName);
                return false;
            }

            var targetType = targetAssembly.GetType(typeName);
            if (targetType == null)
            {
                Debug.LogError("Type not found: " + typeName);
                return false;
            }

            var fieldInfo = targetType.GetField(fieldName, BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance | BindingFlags.Static);
            if (fieldInfo == null)
            {
                Debug.LogError("Field notfound: " + fieldName);
                return false;
            }

            fieldInfo.SetValue(target, value);
            return true;
        }

        public static MethodInfo GetGameObjectSetActiveMethod()
        {
            return typeof(GameObject).GetMethod(
                "SetActive",
                BindingFlags.Instance | BindingFlags.Public,
                null,
                new[] { typeof(bool) },
                null
            );
        }

        public static Component AddComponentToGameObject(GameObject target, string assemblyName, string typeName, bool canUndo = false)
        {
            var targetAssembly = FindAssemblyByName(assemblyName);
            if (targetAssembly == null)
            {
                Debug.LogError("Assembly not found: " + assemblyName);
                return null;
            }
            var targetType = targetAssembly.GetType(typeName);
            if (targetType == null)
            {
                Debug.LogError("Type not found: " + typeName);
                return null;
            }
            
            if (canUndo) return Undo.AddComponent(target, targetType);
            return target.AddComponent(targetType);
        }

        public static bool RegisterVoidUnityEvent(MonoBehaviour eventObject, Type eventType, string eventName, MonoBehaviour methodObject, Type methodType, string methodName)
        {
            // 登録先のイベントの FieldInfo を取得
            var eventFieldInfo = eventType.GetField(
                eventName,
                BindingFlags.Public | BindingFlags.Instance
            );
            if (eventFieldInfo == null)
            {
                Debug.LogError($"Failed to get {eventName} field.");
                return false;
            }

            // フィールド値を取り出す
            var fieldValue = eventFieldInfo.GetValue(eventObject);
            if (fieldValue == null)
            {
                //UnityEventが未初期化の場合、強制的にnew UnityEvent()する。
                var newEvent = new UnityEvent();
                eventFieldInfo.SetValue(eventObject, newEvent); // リフレクションで書き込み
                fieldValue = newEvent;
            }
            var unityEventInstance = fieldValue as UnityEvent;
            if (fieldValue is not UnityEvent unityEvent)
            {
                Debug.LogError($"{eventName} is not UnityEvent.");
                return false;
            }

            // 既に登録されているかチェック
            var cnt = unityEvent.GetPersistentEventCount();

            for (var i = 0; i < cnt; i++)
            {
                var targetObject = unityEvent.GetPersistentTarget(i);
                var targetMethodName = unityEvent.GetPersistentMethodName(i);
                if (targetObject == methodObject && targetMethodName == methodName)
                {
                    EditorUtility.DisplayDialog("Error",
                        "Duplicate UnityEvent listener detected.Cannot add again.",
                        "OK");
                    return false;
                }
            }


            // 登録したいメソッド の MethodInfo を取得
            var methodInfo = methodType.GetMethod(
                methodName,
                BindingFlags.Public | BindingFlags.Instance
            );
            if (methodInfo == null)
            {
                Debug.LogError($"Failed to get {methodName} method.");
                return false;
            }

            var methodDelegate = (UnityAction)Delegate.CreateDelegate(typeof(UnityAction), methodObject, methodInfo);

            UnityEventTools.AddVoidPersistentListener(unityEventInstance, methodDelegate);
            return true;
        }

        public static bool  TryFindAssetPath(string basePath, string searchPattern, out string result)
        {
            result = null;
            try
            {
                string[] directories = Directory.GetDirectories(basePath, "*", SearchOption.AllDirectories);
                foreach (string dir in directories)
                {
                    string potentialPath = Path.Combine(dir, searchPattern);
                    if (File.Exists(potentialPath))
                    {
                        result = potentialPath;
                        return true;
                    }
                }
            }
            catch (Exception)
            {
                return false;
            }
            return false;            
        }
    }
}