using System;
using UnityEngine;

namespace com.nttqonoq.devices.android.mirzalibrary
{
    public class MirzaPlugin : IMirzaPlugin
    {
        private readonly AndroidJavaObject _activity;

        private readonly AndroidJavaObject _library;

        public MirzaPlugin()
        {
            var unityPlayer = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
            _activity = unityPlayer.GetStatic<AndroidJavaObject>("currentActivity");
            _library = new AndroidJavaObject("com.nttqonoq.devices.android.mirzalibrary.MirzaLibrary");
        }

        /// <summary>
        /// モニタリング開始
        /// </summary>
        public void StartMonitoring()
        {
            // コールバックを登録
            _library?.Call("setServiceStateCallback", new ServiceStateCallback(OnServiceStateChanged));
            _library?.Call("setGlassStatusCallback", new GlassStatusCallback(OnGlassStatusChanged));
            _library?.Call("setBatteryLevelCallback", new BatteryLevelCallback(OnBatteryLevelChanged));
            _library?.Call("setSpacesModeStatusCallback", new SpacesModeStatusCallback(OnSpacesModeStatusChanged));
            _library?.Call("setDisplayStatusCallback", new DisplayStatusCallback(OnDisplayStatusChanged));
            _library?.Call("setPowerOffCallback", new PowerOffCallback(OnPowerOffChanged));
            _library?.Call("setGlassTouchGestureStatusCallback",
                new GlassTouchGestureStatusCallback(OnGlassTouchGestureStatusChanged));

            // 連携開始
            _library?.Call("startMonitoring", _activity);
        }

        /// <summary>
        /// モニタリング停止
        /// </summary>
        public void StopMonitoring()
        {
            // 連携解除
            _library?.Call("stopMonitoring", _activity);

            // コールバックを解除
            _library?.Call("setServiceStateCallback", (AndroidJavaObject)null);
            _library?.Call("setGlassStatusCallback", (AndroidJavaObject)null);
            _library?.Call("setBatteryLevelCallback", (AndroidJavaObject)null);
            _library?.Call("setSpacesModeStatusCallback", (AndroidJavaObject)null);
            _library?.Call("setDisplayStatusCallback", (AndroidJavaObject)null);
            _library?.Call("setPowerOffCallback", (AndroidJavaObject)null);
            _library?.Call("setGlassTouchGestureStatusCallback", (AndroidJavaObject)null);
        }

        /// <summary>
        /// ログ出力設定
        /// </summary>
        /// <param name="enabled">有効 or 無効</param>
        public void SetLogEnable(bool enabled)
        {
            _library?.Call("setLogEnable", enabled);
        }

        /// <summary>
        /// バージョン整合性の確認
        /// </summary>
        /// <returns>サービスバージョン</returns>
        public Result<string> CheckVersionConsistency()
        {
            var resultObj = _library?.Call<AndroidJavaObject>("checkVersionConsistency");
            return Result<string>.Convert(resultObj);
        }

        /// <summary>
        /// グラス状態を取得
        /// </summary>
        /// <returns>グラス状態データを含む結果オブジェクト</returns>
        public Result<GlassStatus> GetGlassStatus()
        {
            var resultObj = _library?.Call<AndroidJavaObject>("getGlassStatus");
            return Result<GlassStatus>.Convert(resultObj);
        }

        /// <summary>
        /// グラス状態を取得（非同期）
        /// </summary>
        /// <param name="onResult">取得結果アクション。グラス状態データを含む結果オブジェクトを受け取る。</param>
        public void GetGlassStatusAsync(Action<Result<GlassStatus>> onResult)
        {
            var callbackProxy = new GlassStatusCallback(result => { onResult?.Invoke(result); });
            _library?.Call("getGlassStatusAsync", callbackProxy);
        }

        /// <summary>
        /// 電池残量を取得
        /// </summary>
        /// <returns>電池残量（％）データを含む結果オブジェクト</returns>
        public Result<int> GetBatteryLevel()
        {
            var resultObj = _library?.Call<AndroidJavaObject>("getBatteryLevel");
            return Result<int>.Convert(resultObj);
        }

        /// <summary>
        /// 電池残量を取得（非同期）
        /// </summary>
        /// <param name="onResult">>取得結果アクション。電池残量（％）データを含む結果オブジェクトを受け取る。</param>
        public void GetBatteryLevelAsync(Action<Result<int>> onResult)
        {
            var callbackProxy = new BatteryLevelCallback(result => { onResult?.Invoke(result); });
            _library?.Call("getBatteryLevelAsync", callbackProxy);
        }

        /// <summary>
        /// 充電状態を取得
        /// </summary>
        /// <returns>充電状態データを含む結果オブジェクト</returns>
        public Result<ChargeStatus> GetChargeStatus()
        {
            var resultObj = _library?.Call<AndroidJavaObject>("getChargeStatus");
            return Result<ChargeStatus>.Convert(resultObj);
        }

        /// <summary>
        /// 充電状態を取得（非同期）
        /// </summary>
        /// <param name="onResult">>取得結果アクション。充電状態データを含む結果オブジェクトを受け取る。</param>
        public void GetChargeStatusAsync(Action<Result<ChargeStatus>> onResult)
        {
            var callbackProxy = new ChargeStatusCallback(result => { onResult?.Invoke(result); });
            _library?.Call("getChargeStatusAsync", callbackProxy);
        }

        /// <summary>
        /// Spacesモードを有効化 //TODO: 削除
        /// </summary>
        /// <returns>指示結果オブジェクト。問題がある場合はエラーメッセージを含む。</returns>
        public Result<int> SpacesModeOn()
        {
            var resultObj = _library?.Call<AndroidJavaObject>("spacesModeOn");
            return Result<int>.Convert(resultObj);
        }


        /// <summary>
        /// Spacesモードを有効化（非同期） //TODO: 削除
        /// </summary>
        /// <param name="onResult">>指示結果アクション。結果に問題がある場合はエラーメッセージを含む結果オブジェクトを受け取る。</param>
        public void SpacesModeOnAsync(Action<Result<int>> onResult)
        {
            var callbackProxy = new SpacesModeChangeRequestCallback(result => { onResult?.Invoke(result); });
            _library?.Call("spacesModeOnAsync", callbackProxy);
        }

        /// <summary>
        /// Spacesモードを無効化 
        /// </summary>
        /// <returns>指示結果オブジェクト。問題がある場合はエラーメッセージを含む。</returns>
        public Result<int> SpacesModeOff()
        {
            var resultObj = _library?.Call<AndroidJavaObject>("spacesModeOff");
            return Result<int>.Convert(resultObj);
        }

        /// <summary>
        /// Spacesモードを無効化（非同期）
        /// </summary>
        /// <param name="onResult">>指示結果アクション。結果に問題がある場合はエラーメッセージを含む結果オブジェクトを受け取る。</param>
        public void SpacesModeOffAsync(Action<Result<int>> onResult)
        {
            var callbackProxy = new SpacesModeChangeRequestCallback(result => { onResult?.Invoke(result); });
            _library?.Call("spacesModeOffAsync", callbackProxy);
        }

        /// <summary>
        /// マイク切替を実行する
        /// </summary>
        /// <param name="wearingMicMode">マイクモード1</param>
        /// <param name="frontMicMode">マイクモード2</param>
        /// <param name="mixMode">MIXモードの有無</param>
        /// <returns>指示結果オブジェクト。問題がある場合はエラーメッセージを含む。</returns>
        public Result<SwitchMicrophoneStatus> SwitchMicrophone(MicMode wearingMicMode, MicMode frontMicMode,
            MixMode mixMode)
        {
            var resultObj = _library?.Call<AndroidJavaObject>("switchMicrophone", (int)wearingMicMode, (int)frontMicMode, (int)mixMode);
            return Result<SwitchMicrophoneStatus>.Convert(resultObj);
        }

        /// <summary>
        /// マイク切替を実行する（非同期）
        /// </summary>
        /// <param name="wearingMicMode">マイクモード1</param>
        /// <param name="frontMicMode">マイクモード2</param>
        /// <param name="mixMode">MIXモードの有無</param>
        /// <param name="onResult">指示結果アクション。結果に問題がある場合はエラーメッセージを含む結果オブジェクトを受け取る。</param>
        public void SwitchMicrophoneAsync(MicMode wearingMicMode, MicMode frontMicMode, MixMode mixMode,
            Action<Result<SwitchMicrophoneStatus>> onResult)
        {
            var callbackProxy = new SwitchMicrophoneChangeRequestCallback(result => { onResult?.Invoke(result); });
            _library?.Call("switchMicrophoneAsync", (int)wearingMicMode, (int)frontMicMode, (int)mixMode, callbackProxy);
        }

        /// <summary>
        /// SpacesモードのMiRZAアプリへ遷移する。Spacesモードが無効の場合は遷移しない。
        /// </summary>
        /// <returns>Spaceモード状態データを含む結果オブジェクト</returns>
        public Result<SpacesModeStatus> TransitionToMirzaAppInSpacesMode()
        {
            var unityPlayer = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
            var context = unityPlayer.GetStatic<AndroidJavaObject>("currentActivity");
            var resultObj = _library?.Call<AndroidJavaObject>("transitionToMirzaAppInSpacesMode", context);
            return resultObj == null
                ? new Result<SpacesModeStatus>(State.Unknown, default,null)
                : Result<SpacesModeStatus>.Convert(resultObj);
        }

        /// <summary>
        /// ライブラリバージョンを取得
        /// </summary>
        /// <returns>バージョン</returns>
        public string GetVersion()
        {
            var resultObj = _library?.Call<AndroidJavaObject>("getVersion");
            return resultObj != null ? resultObj.Call<string>("toString") : "";
        }

        /// <summary>
        /// Spacesモード状態を取得
        /// </summary>
        /// <returns>Spacesモード状態データを含む結果オブジェクト</returns>
        public Result<SpacesModeStatus> GetSpacesModeStatus()
        {
            var resultObj = _library?.Call<AndroidJavaObject>("getSpacesModeStatus");
            return Result<SpacesModeStatus>.Convert(resultObj);
        }

        /// <summary>
        /// Spacesモード状態を取得（非同期）
        /// </summary>
        /// <param name="onResult">>取得結果アクション。Spacesモード状態データを含む結果オブジェクトを受け取る。</param>
        public void GetSpacesModeStatusAsync(Action<Result<SpacesModeStatus>> onResult)
        {
            var callbackProxy = new SpacesModeStatusCallback(result => { onResult?.Invoke(result); });
            _library?.Call("getSpacesModeStatusAsync", callbackProxy);
        }


        // サービスステータスのイベントハンドラ
        public event Action<ServiceState> OnServiceStateChanged;

        // グラス状態のイベントハンドラ
        public event Action<GlassStatus> OnGlassStatusChanged;

        // 電池残量のイベントハンドラ
        public event Action<int> OnBatteryLevelChanged;

        // Spacesモードのイベントハンドラ
        public event Action<SpacesModeStatus> OnSpacesModeStatusChanged;

        // 画面表示状態のイベントハンドラ
        public event Action<DisplayStatus> OnDisplayStatusChanged;

        // 手動電源OFFのイベントハンドラ
        public event Action OnPowerOffChanged;

        // グラスタッチパネル操作情報のイベントハンドラ
        public event Action<GlassTouchGestureStatus> OnGlassTouchGestureStatusChanged;
    }
}