using System;
using UnityEngine;

namespace com.nttqonoq.devices.android.mirzalibrary
{
    // ----------------------------------------------------------
    // Javaから呼び出されるコールバックプロキシ
    // ----------------------------------------------------------

    /// <summary>
    /// サービス状態のコールバック
    /// </summary>
    public class ServiceStateCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.ServiceStateCallback";

        private readonly Action<ServiceState> _callback;

        public ServiceStateCallback(Action<ServiceState> callback) : base(JavaInterface)
        {
            _callback = callback;
        }

        public void onServiceStateChanged(AndroidJavaObject javaObject)
        {
            try
            {
                var serviceState = (ServiceState)javaObject.Call<int>("getCode");
                _callback?.Invoke(serviceState);
            }
            catch (Exception e)
            {
                Debug.LogError($"Error in ServiceStateCallback: {e.Message}");
            }
        }
    }

    /// <summary>
    /// グラス状態のコールバック
    /// </summary>
    public class GlassStatusCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.GlassStatusCallback";

        private readonly Action<Result<GlassStatus>> _asyncCallback;

        private readonly Action<GlassStatus> _changeCallback;

        public GlassStatusCallback(Action<GlassStatus> changeCallback) :
            base(JavaInterface)
        {
            _changeCallback = changeCallback;
        }

        public GlassStatusCallback(Action<Result<GlassStatus>> asyncCallback) :
            base(JavaInterface)
        {
            _asyncCallback = asyncCallback;
        }

        public void onGlassStatusChanged(AndroidJavaObject javaObject)
        {
            try
            {
                var glassStatus = GlassStatus.Convert(javaObject);
                _changeCallback?.Invoke(glassStatus);
            }
            catch (Exception e)
            {
                Debug.LogError($"Error in GlassStatusChanged: {e.Message}");
            }
        }

        public void onGlassStatus(AndroidJavaObject javaObject)
        {
            try
            {
                var result = Result<GlassStatus>.Convert(javaObject);
                _asyncCallback?.Invoke(result);
            }
            catch (Exception e)
            {
                Debug.LogError($"Error in GlassStatusAsync: {e.Message}");
            }
        }
    }

    /// <summary>
    /// 電池残量のコールバック
    /// </summary>
    public class BatteryLevelCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.BatteryLevelCallback";

        private readonly Action<Result<int>> _asyncCallback;

        private readonly Action<int> _changeCallback;


        public BatteryLevelCallback(Action<int> changeCallback) : base(JavaInterface)
        {
            _changeCallback = changeCallback;
        }

        public BatteryLevelCallback(Action<Result<int>> asyncCallback) : base(JavaInterface)
        {
            _asyncCallback = asyncCallback;
        }

        public void onBatteryLevelChanged(int level)
        {
            _changeCallback?.Invoke(level);
        }

        public void onBatteryLevel(AndroidJavaObject javaObject)
        {
            var result = Result<int>.Convert(javaObject);
            _asyncCallback?.Invoke(result);
        }
    }

    /// <summary>
    /// 充電状態のコールバック
    /// </summary>
    public class ChargeStatusCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.ChargeStatusCallback";

        private readonly Action<Result<ChargeStatus>> _asyncCallback;

        public ChargeStatusCallback(Action<Result<ChargeStatus>> asyncCallback) : base(JavaInterface)
        {
            _asyncCallback = asyncCallback;
        }

        public void onChargeStatus(AndroidJavaObject javaObject)
        {
            var result = Result<ChargeStatus>.Convert(javaObject);
            _asyncCallback?.Invoke(result);
        }
    }

    /// <summary>
    /// Spacesモードのコールバック
    /// </summary>
    public class SpacesModeStatusCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.SpacesModeStatusCallback";

        private readonly Action<Result<SpacesModeStatus>> _asyncCallback;

        private readonly Action<SpacesModeStatus> _changeCallback;

        public SpacesModeStatusCallback(Action<SpacesModeStatus> changeCallback) : base(JavaInterface)
        {
            _changeCallback = changeCallback;
        }

        public SpacesModeStatusCallback(Action<Result<SpacesModeStatus>> asyncCallback) : base(JavaInterface)
        {
            _asyncCallback = asyncCallback;
        }

        public void onSpacesModeStatusChanged(AndroidJavaObject javaObject)
        {
            var status = (SpacesModeStatus)javaObject.Call<int>("getCode");
            _changeCallback?.Invoke(status);
        }

        public void onSpacesModeStatus(AndroidJavaObject javaObject)
        {
            var result = Result<SpacesModeStatus>.Convert(javaObject);
            _asyncCallback?.Invoke(result);
        }
    }

    /// <summary>
    /// Spacesモード変更指示結果のコールバック
    /// </summary>
    public class SpacesModeChangeRequestCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.SpacesModeChangeRequestCallback";
        
        private readonly Action<Result<int>> _asyncCallback;
        
        public SpacesModeChangeRequestCallback(Action<Result<int>> asyncCallback) : base(JavaInterface)
        {
            _asyncCallback = asyncCallback;
        }
        
        public void onSpacesModeChangeResult(AndroidJavaObject javaObject)
        {
            var result = Result<int>.Convert(javaObject);
            _asyncCallback?.Invoke(result);
        }
    }

    /// <summary>
    /// 画面表示状態のコールバック
    /// </summary>
    public class DisplayStatusCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.DisplayStatusCallback";

        private readonly Action<DisplayStatus> _changeCallback;

        public DisplayStatusCallback(Action<DisplayStatus> changeCallback) : base(JavaInterface)
        {
            _changeCallback = changeCallback;
        }

        public void onDisplayStatusChanged(AndroidJavaObject javaObject)
        {
            var status = (DisplayStatus)javaObject.Call<int>("getCode");
            _changeCallback?.Invoke(status);
        }
    }

    /// <summary>
    /// 手動電源OFFのコールバック
    /// </summary>
    public class PowerOffCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.PowerOffCallback";

        private readonly Action _changeCallback;

        public PowerOffCallback(Action changeCallback) : base(JavaInterface)
        {
            _changeCallback = changeCallback;
        }

        public void onPowerOff()
        {
            _changeCallback?.Invoke();
        }
    }

    /// <summary>
    /// グラスのタッチパネル操作情報コールバック
    /// </summary>
    public class GlassTouchGestureStatusCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.GlassTouchGestureStatusCallback";

        private readonly Action<GlassTouchGestureStatus> _changeCallback;

        public GlassTouchGestureStatusCallback(Action<GlassTouchGestureStatus> changeCallback) : base(JavaInterface)
        {
            _changeCallback = changeCallback;
        }

        public void onGlassTouchGestureStatusChanged(AndroidJavaObject javaObject)
        {
            var status = GlassTouchGestureStatus.Convert(javaObject);
            _changeCallback?.Invoke(status);
        }
    }

    /// <summary>
    /// マイク切替変更指示結果のコールバック
    /// </summary>
    public class SwitchMicrophoneChangeRequestCallback : AndroidJavaProxy
    {
        private const string JavaInterface =
            "com.nttqonoq.devices.android.mirzalibrary.interfaces.SwitchMicrophoneChangeRequestCallback";

        private readonly Action<Result<SwitchMicrophoneStatus>> _asyncCallback;

        public SwitchMicrophoneChangeRequestCallback(Action<Result<SwitchMicrophoneStatus>> asyncCallback) : base(JavaInterface)
        {
            _asyncCallback = asyncCallback;
        }

        public void onSwitchMicrophoneChangeResult(AndroidJavaObject javaObject)
        {
            var result = Result<SwitchMicrophoneStatus>.Convert(javaObject);
            _asyncCallback?.Invoke(result);
        }
    }
}