/*
 * Copyright (C) 2014 The Android Open Source Project
 * Copyright (c) 1996, 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * (C) Copyright Taligent, Inc. 1996, 1997 - All Rights Reserved
 * (C) Copyright IBM Corp. 1996 - 1998 - All Rights Reserved
 *
 *   The original version of this source code and documentation is copyrighted
 * and owned by Taligent, Inc., a wholly-owned subsidiary of IBM. These
 * materials are provided under terms of a License Agreement between Taligent
 * and Sun. This technology is protected by multiple US and International
 * patents. This notice and attribution to Taligent may not be removed.
 *   Taligent is a registered trademark of Taligent, Inc.
 *
 */


package java.text;

import java.util.Locale;
import java.math.RoundingMode;
import java.util.Currency;
import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * {@code DecimalFormat} is a concrete subclass of
 * {@code NumberFormat} that formats decimal numbers. It has a variety of
 * features designed to make it possible to parse and format numbers in any
 * locale, including support for Western, Arabic, and Indic digits.  It also
 * supports different kinds of numbers, including integers (123), fixed-point
 * numbers (123.4), scientific notation (1.23E4), percentages (12%), and
 * currency amounts ($123).  All of these can be localized.
 *
 * <p>To obtain a {@code NumberFormat} for a specific locale, including the
 * default locale, call one of {@code NumberFormat}'s factory methods, such
 * as {@code getInstance()}.  In general, do not call the
 * {@code DecimalFormat} constructors directly, since the
 * {@code NumberFormat} factory methods may return subclasses other than
 * {@code DecimalFormat}. If you need to customize the format object, do
 * something like this:
 *
 * <blockquote><pre>
 * NumberFormat f = NumberFormat.getInstance(loc);
 * if (f instanceof DecimalFormat) {
 *     ((DecimalFormat) f).setDecimalSeparatorAlwaysShown(true);
 * }
 * </pre></blockquote>
 *
 * <p>A {@code DecimalFormat} comprises a <em>pattern</em> and a set of
 * <em>symbols</em>.  The pattern may be set directly using
 * {@code applyPattern()}, or indirectly using the API methods.  The
 * symbols are stored in a {@code DecimalFormatSymbols} object.  When using
 * the {@code NumberFormat} factory methods, the pattern and symbols are
 * read from localized {@code ResourceBundle}s.
 *
 * <h2>Patterns</h2>
 *
 * {@code DecimalFormat} patterns have the following syntax:
 * <blockquote><pre>
 * <i>Pattern:</i>
 *         <i>PositivePattern</i>
 *         <i>PositivePattern</i> ; <i>NegativePattern</i>
 * <i>PositivePattern:</i>
 *         <i>Prefix<sub>opt</sub></i> <i>Number</i> <i>Suffix<sub>opt</sub></i>
 * <i>NegativePattern:</i>
 *         <i>Prefix<sub>opt</sub></i> <i>Number</i> <i>Suffix<sub>opt</sub></i>
 * <i>Prefix:</i>
 *         any Unicode characters except &#92;uFFFE, &#92;uFFFF, and special characters
 * <i>Suffix:</i>
 *         any Unicode characters except &#92;uFFFE, &#92;uFFFF, and special characters
 * <i>Number:</i>
 *         <i>Integer</i> <i>Exponent<sub>opt</sub></i>
 *         <i>Integer</i> . <i>Fraction</i> <i>Exponent<sub>opt</sub></i>
 * <i>Integer:</i>
 *         <i>MinimumInteger</i>
 *         #
 *         # <i>Integer</i>
 *         # , <i>Integer</i>
 * <i>MinimumInteger:</i>
 *         0
 *         0 <i>MinimumInteger</i>
 *         0 , <i>MinimumInteger</i>
 * <i>Fraction:</i>
 *         <i>MinimumFraction<sub>opt</sub></i> <i>OptionalFraction<sub>opt</sub></i>
 * <i>MinimumFraction:</i>
 *         0 <i>MinimumFraction<sub>opt</sub></i>
 * <i>OptionalFraction:</i>
 *         # <i>OptionalFraction<sub>opt</sub></i>
 * <i>Exponent:</i>
 *         E <i>MinimumExponent</i>
 * <i>MinimumExponent:</i>
 *         0 <i>MinimumExponent<sub>opt</sub></i>
 * </pre></blockquote>
 *
 * <p>A {@code DecimalFormat} pattern contains a positive and negative
 * subpattern, for example, {@code "#,##0.00;(#,##0.00)"}.  Each
 * subpattern has a prefix, numeric part, and suffix. The negative subpattern
 * is optional; if absent, then the positive subpattern prefixed with the
 * minus sign ({@code '-' U+002D HYPHEN-MINUS}) is used as the
 * negative subpattern. That is, {@code "0.00"} alone is equivalent to
 * {@code "0.00;-0.00"}.  If there is an explicit negative subpattern, it
 * serves only to specify the negative prefix and suffix; the number of digits,
 * minimal digits, and other characteristics are all the same as the positive
 * pattern. That means that {@code "#,##0.0#;(#)"} produces precisely
 * the same behavior as {@code "#,##0.0#;(#,##0.0#)"}.
 *
 * <p>The prefixes, suffixes, and various symbols used for infinity, digits,
 * grouping separators, decimal separators, etc. may be set to arbitrary
 * values, and they will appear properly during formatting.  However, care must
 * be taken that the symbols and strings do not conflict, or parsing will be
 * unreliable.  For example, either the positive and negative prefixes or the
 * suffixes must be distinct for {@code DecimalFormat.parse()} to be able
 * to distinguish positive from negative values.  (If they are identical, then
 * {@code DecimalFormat} will behave as if no negative subpattern was
 * specified.)  Another example is that the decimal separator and grouping
 * separator should be distinct characters, or parsing will be impossible.
 *
 * <p>The grouping separator is commonly used for thousands, but in some
 * countries it separates ten-thousands. The grouping size is a constant number
 * of digits between the grouping characters, such as 3 for 100,000,000 or 4 for
 * 1,0000,0000.  If you supply a pattern with multiple grouping characters, the
 * interval between the last one and the end of the integer is the one that is
 * used. So {@code "#,##,###,####"} == {@code "######,####"} ==
 * {@code "##,####,####"}.
 *
 * <h3><a id="special_pattern_character">Special Pattern Characters</a></h3>
 *
 * <p>Many characters in a pattern are taken literally; they are matched during
 * parsing and output unchanged during formatting.  Special characters, on the
 * other hand, stand for other characters, strings, or classes of characters.
 * They must be quoted, unless noted otherwise, if they are to appear in the
 * prefix or suffix as literals.
 *
 * <p>The characters listed here are used in non-localized patterns.  Localized
 * patterns use the corresponding characters taken from this formatter's
 * {@code DecimalFormatSymbols} object instead, and these characters lose
 * their special status.  Two exceptions are the currency sign and quote, which
 * are not localized.
 *
 * <blockquote>
 * <table class="striped">
 * <caption style="display:none">Chart showing symbol, location, localized, and meaning.</caption>
 * <thead>
 *     <tr>
 *          <th scope="col" style="text-align:left">Symbol
 *          <th scope="col" style="text-align:left">Location
 *          <th scope="col" style="text-align:left">Localized?
 *          <th scope="col" style="text-align:left">Meaning
 * </thead>
 * <tbody>
 *     <tr style="vertical-align:top">
 *          <th scope="row">{@code 0}
 *          <td>Number
 *          <td>Yes
 *          <td>Digit
 *     <tr style="vertical-align: top">
 *          <th scope="row">{@code #}
 *          <td>Number
 *          <td>Yes
 *          <td>Digit, zero shows as absent
 *     <tr style="vertical-align:top">
 *          <th scope="row">{@code .}
 *          <td>Number
 *          <td>Yes
 *          <td>Decimal separator or monetary decimal separator
 *     <tr style="vertical-align: top">
 *          <th scope="row">{@code -}
 *          <td>Number
 *          <td>Yes
 *          <td>Minus sign
 *     <tr style="vertical-align:top">
 *          <th scope="row">{@code ,}
 *          <td>Number
 *          <td>Yes
 *          <td>Grouping separator or monetary grouping separator
 *     <tr style="vertical-align: top">
 *          <th scope="row">{@code E}
 *          <td>Number
 *          <td>Yes
 *          <td>Separates mantissa and exponent in scientific notation.
 *              <em>Need not be quoted in prefix or suffix.</em>
 *     <tr style="vertical-align:top">
 *          <th scope="row">{@code ;}
 *          <td>Subpattern boundary
 *          <td>Yes
 *          <td>Separates positive and negative subpatterns
 *     <tr style="vertical-align: top">
 *          <th scope="row">{@code %}
 *          <td>Prefix or suffix
 *          <td>Yes
 *          <td>Multiply by 100 and show as percentage
 *     <tr style="vertical-align:top">
 *          <th scope="row">{@code &#92;u2030}
 *          <td>Prefix or suffix
 *          <td>Yes
 *          <td>Multiply by 1000 and show as per mille value
 *     <tr style="vertical-align: top">
 *          <th scope="row">{@code &#164;} ({@code &#92;u00A4})
 *          <td>Prefix or suffix
 *          <td>No
 *          <td>Currency sign, replaced by currency symbol.  If
 *              doubled, replaced by international currency symbol.
 *              If present in a pattern, the monetary decimal/grouping separators
 *              are used instead of the decimal/grouping separators.
 *     <tr style="vertical-align:top">
 *          <th scope="row">{@code '}
 *          <td>Prefix or suffix
 *          <td>No
 *          <td>Used to quote special characters in a prefix or suffix,
 *              for example, {@code "'#'#"} formats 123 to
 *              {@code "#123"}.  To create a single quote
 *              itself, use two in a row: {@code "# o''clock"}.
 * </tbody>
 * </table>
 * </blockquote>
 *
 * <h3>Scientific Notation</h3>
 *
 * <p>Numbers in scientific notation are expressed as the product of a mantissa
 * and a power of ten, for example, 1234 can be expressed as 1.234 x 10^3.  The
 * mantissa is often in the range 1.0 &le; x {@literal <} 10.0, but it need not
 * be.
 * {@code DecimalFormat} can be instructed to format and parse scientific
 * notation <em>only via a pattern</em>; there is currently no factory method
 * that creates a scientific notation format.  In a pattern, the exponent
 * character immediately followed by one or more digit characters indicates
 * scientific notation.  Example: {@code "0.###E0"} formats the number
 * 1234 as {@code "1.234E3"}.
 *
 * <ul>
 * <li>The number of digit characters after the exponent character gives the
 * minimum exponent digit count.  There is no maximum.  Negative exponents are
 * formatted using the localized minus sign, <em>not</em> the prefix and suffix
 * from the pattern.  This allows patterns such as {@code "0.###E0 m/s"}.
 *
 * <li>The minimum and maximum number of integer digits are interpreted
 * together:
 *
 * <ul>
 * <li>If the maximum number of integer digits is greater than their minimum number
 * and greater than 1, it forces the exponent to be a multiple of the maximum
 * number of integer digits, and the minimum number of integer digits to be
 * interpreted as 1.  The most common use of this is to generate
 * <em>engineering notation</em>, in which the exponent is a multiple of three,
 * e.g., {@code "##0.#####E0"}. Using this pattern, the number 12345
 * formats to {@code "12.345E3"}, and 123456 formats to
 * {@code "123.456E3"}.
 *
 * <li>Otherwise, the minimum number of integer digits is achieved by adjusting the
 * exponent.  Example: 0.00123 formatted with {@code "00.###E0"} yields
 * {@code "12.3E-4"}.
 * </ul>
 *
 * <li>The number of significant digits in the mantissa is the sum of the
 * <em>minimum integer</em> and <em>maximum fraction</em> digits, and is
 * unaffected by the maximum integer digits.  For example, 12345 formatted with
 * {@code "##0.##E0"} is {@code "12.3E3"}. To show all digits, set
 * the significant digits count to zero.  The number of significant digits
 * does not affect parsing.
 *
 * <li>Exponential patterns may not contain grouping separators.
 * </ul>
 *
 * <h3>Rounding</h3>
 *
 * {@code DecimalFormat} provides rounding modes defined in
 * {@link java.math.RoundingMode} for formatting.  By default, it uses
 * {@link java.math.RoundingMode#HALF_EVEN RoundingMode.HALF_EVEN}.
 *
 * <h3>Digits</h3>
 *
 * For formatting, {@code DecimalFormat} uses the ten consecutive
 * characters starting with the localized zero digit defined in the
 * {@code DecimalFormatSymbols} object as digits. For parsing, these
 * digits as well as all Unicode decimal digits, as defined by
 * {@link java.lang.Character#digit Character.digit}, are recognized.
 *
 * <h4>Special Values</h4>
 *
 * <p>{@code NaN} is formatted as a string, which typically has a single character
 * {@code &#92;uFFFD}.  This string is determined by the
 * {@code DecimalFormatSymbols} object.  This is the only value for which
 * the prefixes and suffixes are not used.
 *
 * <p>Infinity is formatted as a string, which typically has a single character
 * {@code &#92;u221E}, with the positive or negative prefixes and suffixes
 * applied.  The infinity string is determined by the
 * {@code DecimalFormatSymbols} object.
 *
 * <p>Negative zero ({@code "-0"}) parses to
 * <ul>
 * <li>{@code BigDecimal(0)} if {@code isParseBigDecimal()} is
 * true,
 * <li>{@code Long(0)} if {@code isParseBigDecimal()} is false
 *     and {@code isParseIntegerOnly()} is true,
 * <li>{@code Double(-0.0)} if both {@code isParseBigDecimal()}
 * and {@code isParseIntegerOnly()} are false.
 * </ul>
 *
 * <h3><a id="synchronization">Synchronization</a></h3>
 *
 * <p>
 * Decimal formats are generally not synchronized.
 * It is recommended to create separate format instances for each thread.
 * If multiple threads access a format concurrently, it must be synchronized
 * externally.
 *
 * <h3>Example</h3>
 *
 * <blockquote><pre><strong>{@code
 * // Print out a number using the localized number, integer, currency,
 * // and percent format for each locale}</strong>{@code
 * Locale[] locales = NumberFormat.getAvailableLocales();
 * double myNumber = -1234.56;
 * NumberFormat form;
 * for (int j = 0; j < 4; ++j) {
 *     System.out.println("FORMAT");
 *     for (int i = 0; i < locales.length; ++i) {
 *         if (locales[i].getCountry().length() == 0) {
 *            continue; // Skip language-only locales
 *         }
 *         System.out.print(locales[i].getDisplayName());
 *         switch (j) {
 *         case 0:
 *             form = NumberFormat.getInstance(locales[i]); break;
 *         case 1:
 *             form = NumberFormat.getIntegerInstance(locales[i]); break;
 *         case 2:
 *             form = NumberFormat.getCurrencyInstance(locales[i]); break;
 *         default:
 *             form = NumberFormat.getPercentInstance(locales[i]); break;
 *         }
 *         if (form instanceof DecimalFormat) {
 *             System.out.print(": " + ((DecimalFormat) form).toPattern());
 *         }
 *         System.out.print(" -> " + form.format(myNumber));
 *         try {
 *             System.out.println(" -> " + form.parse(form.format(myNumber)));
 *         } catch (ParseException e) {}
 *     }
 * }
 * }</pre></blockquote>
 *
 * @see          <a href="http://docs.oracle.com/javase/tutorial/i18n/format/decimalFormat.html">Java Tutorial</a>
 * @see          java.text.NumberFormat
 * @see          java.text.DecimalFormatSymbols
 * @see          java.text.ParsePosition
 * @author       Mark Davis
 * @author       Alan Liu
 * @since 1.1
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class DecimalFormat extends java.text.NumberFormat {

/**
 * Creates a DecimalFormat using the default pattern and symbols
 * for the default {@link java.util.Locale.Category#FORMAT FORMAT} locale.
 * This is a convenient way to obtain a
 * DecimalFormat when internationalization is not the main concern.
 * <p>
 * To obtain standard formats for a given locale, use the factory methods
 * on NumberFormat such as getNumberInstance. These factories will
 * return the most appropriate sub-class of NumberFormat for a given
 * locale.
 *
 * @see java.text.NumberFormat#getInstance
 * @see java.text.NumberFormat#getNumberInstance
 * @see java.text.NumberFormat#getCurrencyInstance
 * @see java.text.NumberFormat#getPercentInstance
 */

public DecimalFormat() { throw new RuntimeException("Stub!"); }

/**
 * Creates a DecimalFormat using the given pattern and the symbols
 * for the default {@link java.util.Locale.Category#FORMAT FORMAT} locale.
 * This is a convenient way to obtain a
 * DecimalFormat when internationalization is not the main concern.
 * <p>
 * To obtain standard formats for a given locale, use the factory methods
 * on NumberFormat such as getNumberInstance. These factories will
 * return the most appropriate sub-class of NumberFormat for a given
 * locale.
 *
 * @param pattern a non-localized pattern string.
 * @throws    java.lang.NullPointerException if {@code pattern} is null
 * @throws    java.lang.IllegalArgumentException if the given pattern is invalid.
 * @see java.text.NumberFormat#getInstance
 * @see java.text.NumberFormat#getNumberInstance
 * @see java.text.NumberFormat#getCurrencyInstance
 * @see java.text.NumberFormat#getPercentInstance
 */

public DecimalFormat(java.lang.String pattern) { throw new RuntimeException("Stub!"); }

/**
 * Creates a DecimalFormat using the given pattern and symbols.
 * Use this constructor when you need to completely customize the
 * behavior of the format.
 * <p>
 * To obtain standard formats for a given
 * locale, use the factory methods on NumberFormat such as
 * getInstance or getCurrencyInstance. If you need only minor adjustments
 * to a standard format, you can modify the format returned by
 * a NumberFormat factory method.
 *
 * @param pattern a non-localized pattern string
 * @param symbols the set of symbols to be used
 * @throws    java.lang.NullPointerException if any of the given arguments is null
 * @throws    java.lang.IllegalArgumentException if the given pattern is invalid
 * @see java.text.NumberFormat#getInstance
 * @see java.text.NumberFormat#getNumberInstance
 * @see java.text.NumberFormat#getCurrencyInstance
 * @see java.text.NumberFormat#getPercentInstance
 * @see java.text.DecimalFormatSymbols
 */

public DecimalFormat(java.lang.String pattern, java.text.DecimalFormatSymbols symbols) { throw new RuntimeException("Stub!"); }

/**
 * Formats a number and appends the resulting text to the given string
 * buffer.
 * The number can be of any subclass of {@link java.lang.Number}.
 * <p>
 * This implementation uses the maximum precision permitted.
 * @param number     the number to format
 * @param toAppendTo the {@code StringBuffer} to which the formatted
 *                   text is to be appended
 * @param pos        keeps track on the position of the field within the
 *                   returned string. For example, for formatting a number
 *                   {@code 1234567.89} in {@code Locale.US} locale,
 *                   if the given {@code fieldPosition} is
 *                   {@link java.text.NumberFormat#INTEGER_FIELD NumberFormat#INTEGER_FIELD}, the begin index
 *                   and end index of {@code fieldPosition} will be set
 *                   to 0 and 9, respectively for the output string
 *                   {@code 1,234,567.89}.
 * @return           the value passed in as {@code toAppendTo}
 * @throws           java.lang.IllegalArgumentException if {@code number} is
 *                   null or not an instance of {@code Number}.
 * @throws           java.lang.NullPointerException if {@code toAppendTo} or
 *                   {@code pos} is null
 * @throws           java.lang.ArithmeticException if rounding is needed with rounding
 *                   mode being set to RoundingMode.UNNECESSARY
 * @see              java.text.FieldPosition
 */

public final java.lang.StringBuffer format(java.lang.Object number, java.lang.StringBuffer toAppendTo, java.text.FieldPosition pos) { throw new RuntimeException("Stub!"); }

/**
 * Formats a double to produce a string.
 * @param number    The double to format
 * @param result    where the text is to be appended
 * @param fieldPosition    keeps track on the position of the field within
 *                         the returned string. For example, for formatting
 *                         a number {@code 1234567.89} in {@code Locale.US}
 *                         locale, if the given {@code fieldPosition} is
 *                         {@link java.text.NumberFormat#INTEGER_FIELD NumberFormat#INTEGER_FIELD}, the begin index
 *                         and end index of {@code fieldPosition} will be set
 *                         to 0 and 9, respectively for the output string
 *                         {@code 1,234,567.89}.
 * @throws    java.lang.NullPointerException if {@code result} or
 *            {@code fieldPosition} is {@code null}
 * @throws    java.lang.ArithmeticException if rounding is needed with rounding
 *            mode being set to RoundingMode.UNNECESSARY
 * @return The formatted number string
 * @see java.text.FieldPosition
 */

public java.lang.StringBuffer format(double number, java.lang.StringBuffer result, java.text.FieldPosition fieldPosition) { throw new RuntimeException("Stub!"); }

/**
 * Format a long to produce a string.
 * @param number    The long to format
 * @param result    where the text is to be appended
 * @param fieldPosition    keeps track on the position of the field within
 *                         the returned string. For example, for formatting
 *                         a number {@code 123456789} in {@code Locale.US}
 *                         locale, if the given {@code fieldPosition} is
 *                         {@link java.text.NumberFormat#INTEGER_FIELD NumberFormat#INTEGER_FIELD}, the begin index
 *                         and end index of {@code fieldPosition} will be set
 *                         to 0 and 11, respectively for the output string
 *                         {@code 123,456,789}.
 * @throws          java.lang.NullPointerException if {@code result} or
 *                  {@code fieldPosition} is {@code null}
 * @throws          java.lang.ArithmeticException if rounding is needed with rounding
 *                  mode being set to RoundingMode.UNNECESSARY
 * @return The formatted number string
 * @see java.text.FieldPosition
 */

public java.lang.StringBuffer format(long number, java.lang.StringBuffer result, java.text.FieldPosition fieldPosition) { throw new RuntimeException("Stub!"); }

/**
 * Formats an Object producing an {@code AttributedCharacterIterator}.
 * You can use the returned {@code AttributedCharacterIterator}
 * to build the resulting String, as well as to determine information
 * about the resulting String.
 * <p>
 * Each attribute key of the AttributedCharacterIterator will be of type
 * {@code NumberFormat.Field}, with the attribute value being the
 * same as the attribute key.
 *
 * @throws    java.lang.NullPointerException if obj is null.
 * @throws    java.lang.IllegalArgumentException when the Format cannot format the
 *            given object.
 * @throws           java.lang.ArithmeticException if rounding is needed with rounding
 *                   mode being set to RoundingMode.UNNECESSARY
 * @param obj The object to format
 * @return AttributedCharacterIterator describing the formatted value.
 * @since 1.4
 */

public java.text.AttributedCharacterIterator formatToCharacterIterator(java.lang.Object obj) { throw new RuntimeException("Stub!"); }

/**
 * Parses text from a string to produce a {@code Number}.
 * <p>
 * The method attempts to parse text starting at the index given by
 * {@code pos}.
 * If parsing succeeds, then the index of {@code pos} is updated
 * to the index after the last character used (parsing does not necessarily
 * use all characters up to the end of the string), and the parsed
 * number is returned. The updated {@code pos} can be used to
 * indicate the starting point for the next call to this method.
 * If an error occurs, then the index of {@code pos} is not
 * changed, the error index of {@code pos} is set to the index of
 * the character where the error occurred, and null is returned.
 * <p>
 * The subclass returned depends on the value of {@link #isParseBigDecimal}
 * as well as on the string being parsed.
 * <ul>
 *   <li>If {@code isParseBigDecimal()} is false (the default),
 *       most integer values are returned as {@code Long}
 *       objects, no matter how they are written: {@code "17"} and
 *       {@code "17.000"} both parse to {@code Long(17)}.
 *       Values that cannot fit into a {@code Long} are returned as
 *       {@code Double}s. This includes values with a fractional part,
 *       infinite values, {@code NaN}, and the value -0.0.
 *       {@code DecimalFormat} does <em>not</em> decide whether to
 *       return a {@code Double} or a {@code Long} based on the
 *       presence of a decimal separator in the source string. Doing so
 *       would prevent integers that overflow the mantissa of a double,
 *       such as {@code "-9,223,372,036,854,775,808.00"}, from being
 *       parsed accurately.
 *       <p>
 *       Callers may use the {@code Number} methods
 *       {@code doubleValue}, {@code longValue}, etc., to obtain
 *       the type they want.
 *   <li>If {@code isParseBigDecimal()} is true, values are returned
 *       as {@code BigDecimal} objects. The values are the ones
 *       constructed by {@link java.math.BigDecimal#BigDecimal(String)}
 *       for corresponding strings in locale-independent format. The
 *       special cases negative and positive infinity and NaN are returned
 *       as {@code Double} instances holding the values of the
 *       corresponding {@code Double} constants.
 * </ul>
 * <p>
 * {@code DecimalFormat} parses all Unicode characters that represent
 * decimal digits, as defined by {@code Character.digit()}. In
 * addition, {@code DecimalFormat} also recognizes as digits the ten
 * consecutive characters starting with the localized zero digit defined in
 * the {@code DecimalFormatSymbols} object.
 *
 * @param text the string to be parsed
 * @param pos  A {@code ParsePosition} object with index and error
 *             index information as described above.
 * @return     the parsed value, or {@code null} if the parse fails
 * @throws     java.lang.NullPointerException if {@code text} or
 *             {@code pos} is null.
 */

public java.lang.Number parse(java.lang.String text, java.text.ParsePosition pos) { throw new RuntimeException("Stub!"); }

/**
 * Returns a copy of the decimal format symbols, which is generally not
 * changed by the programmer or user.
 * @return a copy of the desired DecimalFormatSymbols
 * @see java.text.DecimalFormatSymbols
 */

public java.text.DecimalFormatSymbols getDecimalFormatSymbols() { throw new RuntimeException("Stub!"); }

/**
 * Sets the decimal format symbols, which is generally not changed
 * by the programmer or user.
 * @param newSymbols desired DecimalFormatSymbols
 * @see java.text.DecimalFormatSymbols
 */

public void setDecimalFormatSymbols(java.text.DecimalFormatSymbols newSymbols) { throw new RuntimeException("Stub!"); }

/**
 * Get the positive prefix.
 * <P>Examples: +123, $123, sFr123
 *
 * @return the positive prefix
 */

public java.lang.String getPositivePrefix() { throw new RuntimeException("Stub!"); }

/**
 * Set the positive prefix.
 * <P>Examples: +123, $123, sFr123
 *
 * @param newValue the new positive prefix
 */

public void setPositivePrefix(java.lang.String newValue) { throw new RuntimeException("Stub!"); }

/**
 * Get the negative prefix.
 * <P>Examples: -123, ($123) (with negative suffix), sFr-123
 *
 * @return the negative prefix
 */

public java.lang.String getNegativePrefix() { throw new RuntimeException("Stub!"); }

/**
 * Set the negative prefix.
 * <P>Examples: -123, ($123) (with negative suffix), sFr-123
 *
 * @param newValue the new negative prefix
 */

public void setNegativePrefix(java.lang.String newValue) { throw new RuntimeException("Stub!"); }

/**
 * Get the positive suffix.
 * <P>Example: 123%
 *
 * @return the positive suffix
 */

public java.lang.String getPositiveSuffix() { throw new RuntimeException("Stub!"); }

/**
 * Set the positive suffix.
 * <P>Example: 123%
 *
 * @param newValue the new positive suffix
 */

public void setPositiveSuffix(java.lang.String newValue) { throw new RuntimeException("Stub!"); }

/**
 * Get the negative suffix.
 * <P>Examples: -123%, ($123) (with positive suffixes)
 *
 * @return the negative suffix
 */

public java.lang.String getNegativeSuffix() { throw new RuntimeException("Stub!"); }

/**
 * Set the negative suffix.
 * <P>Examples: 123%
 *
 * @param newValue the new negative suffix
 */

public void setNegativeSuffix(java.lang.String newValue) { throw new RuntimeException("Stub!"); }

/**
 * Gets the multiplier for use in percent, per mille, and similar
 * formats.
 *
 * @return the multiplier
 * @see #setMultiplier(int)
 */

public int getMultiplier() { throw new RuntimeException("Stub!"); }

/**
 * Sets the multiplier for use in percent, per mille, and similar
 * formats.
 * For a percent format, set the multiplier to 100 and the suffixes to
 * have '%' (for Arabic, use the Arabic percent sign).
 * For a per mille format, set the multiplier to 1000 and the suffixes to
 * have '&#92;u2030'.
 *
 * <P>Example: with multiplier 100, 1.23 is formatted as "123", and
 * "123" is parsed into 1.23.
 *
 * @param newValue the new multiplier
 * @see #getMultiplier
 */

public void setMultiplier(int newValue) { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public void setGroupingUsed(boolean newValue) { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public boolean isGroupingUsed() { throw new RuntimeException("Stub!"); }

/**
 * Return the grouping size. Grouping size is the number of digits between
 * grouping separators in the integer portion of a number.  For example,
 * in the number "123,456.78", the grouping size is 3. Grouping size of
 * zero designates that grouping is not used, which provides the same
 * formatting as if calling {@link #setGroupingUsed(boolean)
 * setGroupingUsed(false)}.
 *
 * @return the grouping size
 * @see #setGroupingSize
 * @see java.text.NumberFormat#isGroupingUsed
 * @see java.text.DecimalFormatSymbols#getGroupingSeparator
 */

public int getGroupingSize() { throw new RuntimeException("Stub!"); }

/**
 * Set the grouping size. Grouping size is the number of digits between
 * grouping separators in the integer portion of a number.  For example,
 * in the number "123,456.78", the grouping size is 3. Grouping size of
 * zero designates that grouping is not used, which provides the same
 * formatting as if calling {@link #setGroupingUsed(boolean)
 * setGroupingUsed(false)}.
 * <p>
 * The value passed in is converted to a byte, which may lose information.
 * Values that are negative or greater than
 * {@link java.lang.Byte#MAX_VALUE Byte.MAX_VALUE}, will throw an
 * {@code IllegalArgumentException}.
 *
 * @param newValue the new grouping size
 * @see #getGroupingSize
 * @see java.text.NumberFormat#setGroupingUsed
 * @see java.text.DecimalFormatSymbols#setGroupingSeparator
 * @throws java.lang.IllegalArgumentException if {@code newValue} is negative or
 *          greater than {@link java.lang.Byte#MAX_VALUE Byte.MAX_VALUE}
 */

public void setGroupingSize(int newValue) { throw new RuntimeException("Stub!"); }

/**
 * Allows you to get the behavior of the decimal separator with integers.
 * (The decimal separator will always appear with decimals.)
 * <P>Example: Decimal ON: 12345 &rarr; 12345.; OFF: 12345 &rarr; 12345
 *
 * @return {@code true} if the decimal separator is always shown;
 *         {@code false} otherwise
 */

public boolean isDecimalSeparatorAlwaysShown() { throw new RuntimeException("Stub!"); }

/**
 * Allows you to set the behavior of the decimal separator with integers.
 * (The decimal separator will always appear with decimals.)
 * <P>Example: Decimal ON: 12345 &rarr; 12345.; OFF: 12345 &rarr; 12345
 *
 * @param newValue {@code true} if the decimal separator is always shown;
 *                 {@code false} otherwise
 */

public void setDecimalSeparatorAlwaysShown(boolean newValue) { throw new RuntimeException("Stub!"); }

/**
 * Returns whether the {@link #parse(java.lang.String,java.text.ParsePosition)}
 * method returns {@code BigDecimal}. The default value is false.
 *
 * @return {@code true} if the parse method returns BigDecimal;
 *         {@code false} otherwise
 * @see #setParseBigDecimal
 * @since 1.5
 */

public boolean isParseBigDecimal() { throw new RuntimeException("Stub!"); }

/**
 * Sets whether the {@link #parse(java.lang.String,java.text.ParsePosition)}
 * method returns {@code BigDecimal}.
 *
 * @param newValue {@code true} if the parse method returns BigDecimal;
 *                 {@code false} otherwise
 * @see #isParseBigDecimal
 * @since 1.5
 */

public void setParseBigDecimal(boolean newValue) { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public boolean isParseIntegerOnly() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public void setParseIntegerOnly(boolean value) { throw new RuntimeException("Stub!"); }

/**
 * Standard override; no change in semantics.
 */

public java.lang.Object clone() { throw new RuntimeException("Stub!"); }

/**
 * Overrides equals
 */

public boolean equals(java.lang.Object obj) { throw new RuntimeException("Stub!"); }

/**
 * Overrides hashCode
 */

public int hashCode() { throw new RuntimeException("Stub!"); }

/**
 * Synthesizes a pattern string that represents the current state
 * of this Format object.
 *
 * @return a pattern string
 * @see #applyPattern
 */

public java.lang.String toPattern() { throw new RuntimeException("Stub!"); }

/**
 * Synthesizes a localized pattern string that represents the current
 * state of this Format object.
 *
 * @return a localized pattern string
 * @see #applyPattern
 */

public java.lang.String toLocalizedPattern() { throw new RuntimeException("Stub!"); }

/**
 * Apply the given pattern to this Format object.  A pattern is a
 * short-hand specification for the various formatting properties.
 * These properties can also be changed individually through the
 * various setter methods.
 * <p>
 * There is no limit to integer digits set
 * by this routine, since that is the typical end-user desire;
 * use setMaximumInteger if you want to set a real value.
 * For negative numbers, use a second pattern, separated by a semicolon
 * <P>Example {@code "#,#00.0#"} &rarr; 1,234.56
 * <P>This means a minimum of 2 integer digits, 1 fraction digit, and
 * a maximum of 2 fraction digits.
 * <p>Example: {@code "#,#00.0#;(#,#00.0#)"} for negatives in
 * parentheses.
 * <p>In negative patterns, the minimum and maximum counts are ignored;
 * these are presumed to be set in the positive pattern.
 *
 * @param pattern a new pattern
 * @throws    java.lang.NullPointerException if {@code pattern} is null
 * @throws    java.lang.IllegalArgumentException if the given pattern is invalid.
 */

public void applyPattern(java.lang.String pattern) { throw new RuntimeException("Stub!"); }

/**
 * Apply the given pattern to this Format object.  The pattern
 * is assumed to be in a localized notation. A pattern is a
 * short-hand specification for the various formatting properties.
 * These properties can also be changed individually through the
 * various setter methods.
 * <p>
 * There is no limit to integer digits set
 * by this routine, since that is the typical end-user desire;
 * use setMaximumInteger if you want to set a real value.
 * For negative numbers, use a second pattern, separated by a semicolon
 * <P>Example {@code "#,#00.0#"} &rarr; 1,234.56
 * <P>This means a minimum of 2 integer digits, 1 fraction digit, and
 * a maximum of 2 fraction digits.
 * <p>Example: {@code "#,#00.0#;(#,#00.0#)"} for negatives in
 * parentheses.
 * <p>In negative patterns, the minimum and maximum counts are ignored;
 * these are presumed to be set in the positive pattern.
 *
 * @param pattern a new pattern
 * @throws    java.lang.NullPointerException if {@code pattern} is null
 * @throws    java.lang.IllegalArgumentException if the given pattern is invalid.
 */

public void applyLocalizedPattern(java.lang.String pattern) { throw new RuntimeException("Stub!"); }

/**
 * Sets the maximum number of digits allowed in the integer portion of a
 * number.
 * For formatting numbers other than {@code BigInteger} and
 * {@code BigDecimal} objects, the lower of {@code newValue} and
 * 309 is used. Negative input values are replaced with 0.
 * @see java.text.NumberFormat#setMaximumIntegerDigits
 */

public void setMaximumIntegerDigits(int newValue) { throw new RuntimeException("Stub!"); }

/**
 * Sets the minimum number of digits allowed in the integer portion of a
 * number.
 * For formatting numbers other than {@code BigInteger} and
 * {@code BigDecimal} objects, the lower of {@code newValue} and
 * 309 is used. Negative input values are replaced with 0.
 * @see java.text.NumberFormat#setMinimumIntegerDigits
 */

public void setMinimumIntegerDigits(int newValue) { throw new RuntimeException("Stub!"); }

/**
 * Sets the maximum number of digits allowed in the fraction portion of a
 * number.
 * For formatting numbers other than {@code BigInteger} and
 * {@code BigDecimal} objects, the lower of {@code newValue} and
 * 340 is used. Negative input values are replaced with 0.
 * @see java.text.NumberFormat#setMaximumFractionDigits
 */

public void setMaximumFractionDigits(int newValue) { throw new RuntimeException("Stub!"); }

/**
 * Sets the minimum number of digits allowed in the fraction portion of a
 * number.
 * For formatting numbers other than {@code BigInteger} and
 * {@code BigDecimal} objects, the lower of {@code newValue} and
 * 340 is used. Negative input values are replaced with 0.
 * @see java.text.NumberFormat#setMinimumFractionDigits
 */

public void setMinimumFractionDigits(int newValue) { throw new RuntimeException("Stub!"); }

/**
 * Gets the maximum number of digits allowed in the integer portion of a
 * number.
 * For formatting numbers other than {@code BigInteger} and
 * {@code BigDecimal} objects, the lower of the return value and
 * 309 is used.
 * @see #setMaximumIntegerDigits
 */

public int getMaximumIntegerDigits() { throw new RuntimeException("Stub!"); }

/**
 * Gets the minimum number of digits allowed in the integer portion of a
 * number.
 * For formatting numbers other than {@code BigInteger} and
 * {@code BigDecimal} objects, the lower of the return value and
 * 309 is used.
 * @see #setMinimumIntegerDigits
 */

public int getMinimumIntegerDigits() { throw new RuntimeException("Stub!"); }

/**
 * Gets the maximum number of digits allowed in the fraction portion of a
 * number.
 * For formatting numbers other than {@code BigInteger} and
 * {@code BigDecimal} objects, the lower of the return value and
 * 340 is used.
 * @see #setMaximumFractionDigits
 */

public int getMaximumFractionDigits() { throw new RuntimeException("Stub!"); }

/**
 * Gets the minimum number of digits allowed in the fraction portion of a
 * number.
 * For formatting numbers other than {@code BigInteger} and
 * {@code BigDecimal} objects, the lower of the return value and
 * 340 is used.
 * @see #setMinimumFractionDigits
 */

public int getMinimumFractionDigits() { throw new RuntimeException("Stub!"); }

/**
 * Gets the currency used by this decimal format when formatting
 * currency values.
 * The currency is obtained by calling
 * {@link java.text.DecimalFormatSymbols#getCurrency DecimalFormatSymbols.getCurrency}
 * on this number format's symbols.
 *
 * @return the currency used by this decimal format, or {@code null}
 * @since 1.4
 */

public java.util.Currency getCurrency() { throw new RuntimeException("Stub!"); }

/**
 * Sets the currency used by this number format when formatting
 * currency values. This does not update the minimum or maximum
 * number of fraction digits used by the number format.
 * The currency is set by calling
 * {@link java.text.DecimalFormatSymbols#setCurrency DecimalFormatSymbols.setCurrency}
 * on this number format's symbols.
 *
 * @param currency the new currency to be used by this decimal format
 * @throws    java.lang.NullPointerException if {@code currency} is null
 * @since 1.4
 */

public void setCurrency(java.util.Currency currency) { throw new RuntimeException("Stub!"); }

/**
 * Gets the {@link java.math.RoundingMode} used in this DecimalFormat.
 *
 * @return The {@code RoundingMode} used for this DecimalFormat.
 * @see #setRoundingMode(RoundingMode)
 * @since 1.6
 */

public java.math.RoundingMode getRoundingMode() { throw new RuntimeException("Stub!"); }

/**
 * Sets the {@link java.math.RoundingMode} used in this DecimalFormat.
 *
 * @param roundingMode The {@code RoundingMode} to be used
 * @see #getRoundingMode()
 * @throws    java.lang.NullPointerException if {@code roundingMode} is null.
 * @since 1.6
 */

public void setRoundingMode(java.math.RoundingMode roundingMode) { throw new RuntimeException("Stub!"); }
}

