// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.


package android.net.http;

import java.io.IOException;

/**
 * Abstract class allowing the embedder to provide an upload body to {@link android.net.http.UrlRequest UrlRequest}. It supports
 * both non-chunked (size known in advanced) and chunked (size not known in advance) uploads. Be
 * aware that not all servers support chunked uploads.
 *
 * <p>An upload is either always chunked, across multiple uploads if the data
 * ends up being sent more than once, or never chunked.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract class UploadDataProvider implements java.io.Closeable {

public UploadDataProvider() { throw new RuntimeException("Stub!"); }

/**
 * If this is a non-chunked upload, returns the length of the upload. Must always return -1 if
 * this is a chunked upload.
 *
 * @return the length of the upload for non-chunked uploads, -1 otherwise.
 * @throws java.io.IOException if any IOException occurred during the process.
 */

public abstract long getLength() throws java.io.IOException;

/**
 * Reads upload data into {@code byteBuffer}. Upon completion, the buffer's position is updated
 * to the end of the bytes that were read. The buffer's limit is not changed. Each call of this
 * method must be followed be a single call, either synchronous or asynchronous, to {@code
 * uploadDataSink}: {@link android.net.http.UploadDataSink#onReadSucceeded UploadDataSink#onReadSucceeded} on success or {@link android.net.http.UploadDataSink#onReadError  } on failure. Neither read nor rewind will be called until one of
 * those methods or the other is called. Even if the associated {@link android.net.http.UrlRequest UrlRequest} is canceled,
 * one or the other must still be called before resources can be safely freed. Throwing an
 * exception will also result in resources being freed and the request being errored out.
 *
 * @param uploadDataSink The object to notify when the read has completed, successfully or
 * otherwise.
 * @param byteBuffer The buffer to copy the read bytes into. Do not change byteBuffer's limit.
 * @throws java.io.IOException if any IOException occurred during the process. {@link android.net.http.UrlRequest.Callback#onFailed  } will be called with the thrown exception set as the cause of
 * the
 * {@link android.net.http.CallbackException CallbackException}.
 */

public abstract void read(@android.annotation.NonNull android.net.http.UploadDataSink uploadDataSink, @android.annotation.NonNull java.nio.ByteBuffer byteBuffer) throws java.io.IOException;

/**
 * Rewinds upload data. Each call must be followed be a single call, either synchronous or
 * asynchronous, to {@code uploadDataSink}: {@link android.net.http.UploadDataSink#onRewindSucceeded UploadDataSink#onRewindSucceeded} on success
 * or
 * {@link android.net.http.UploadDataSink#onRewindError UploadDataSink#onRewindError} on failure. Neither read nor rewind will be called until
 * one of those methods or the other is called. Even if the associated {@link android.net.http.UrlRequest UrlRequest} is
 * canceled, one or the other must still be called before resources can be safely freed.
 * Throwing an exception will also result in resources being freed and the request being errored
 * out.
 *
 * <p>If rewinding is not supported, this should call
 * {@link android.net.http.UploadDataSink#onRewindError UploadDataSink#onRewindError}. Note that rewinding is required to follow redirects
 * that preserve the upload body, and for retrying when the server times out stale sockets.
 *
 * @param uploadDataSink The object to notify when the rewind operation has completed,
 * successfully or otherwise.
 * @throws java.io.IOException if any IOException occurred during the process. {@link android.net.http.UrlRequest.Callback#onFailed  } will be called with the thrown exception set as the cause of
 * the
 * {@link android.net.http.CallbackException CallbackException}.
 */

public abstract void rewind(@android.annotation.NonNull android.net.http.UploadDataSink uploadDataSink) throws java.io.IOException;

/**
 * Called when this UploadDataProvider is no longer needed by a request, so that resources (like
 * a file) can be explicitly released.
 *
 * @throws java.io.IOException if any IOException occurred during the process. This will cause the
 *         request
 * to fail if it is not yet complete; otherwise it will be logged.
 */

public void close() throws java.io.IOException { throw new RuntimeException("Stub!"); }
}

